// $Id: SEIS16v2.cpp,v 1.6 2019/03/29 22:06:55 pm Exp $          
/* |------------------------------------------------------------|
   |SEIS16v2.cpp  Seismic Recorder called by TK Program         |
   |Author: Dr. P. Michaels, PE <pmsolid@cableone.net>          |
   |or <paulmichaels@boisestate.edu>                            |
   |  NOTICE:                                                   |
   |  Copyright (C) 2019 Paul Michaels                          |
   |  This program is free software; you can                    |
   |  redistribute it and/or modify it under the terms          |
   |  of the GNU General Public License as published            |
   |  by the Free Software Foundation; either version           |
   |  3 of the License, or (at your option) any later           |
   |  version.  This program is distributed in the              |
   |  hope that it will be useful, but                          |
   |  WITHOUT ANY WARRANTY; without even the implied            |
   |  warranty of MERCHANTABILITY or FITNESS FOR A              |
   |  PARTICULAR PURPOSE.  See the GNU General Public           |
   |  License for more details.                                 |
   |  You should have received a copy of the GNU                |
   |  General Public License along with this program;           |
   |  if not, write to the Free Software Foundation,            |
   |  Inc., 675 Mass Ave, Cambridge, MA 02139, USA.             |
   |------------------------------------------------------------|
 */
// UNIPOLAR VERSION
#include <stdio.h>
#include <termios.h>
#include <cstdio>
#include <cstring>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>
#include <float.h>
#include <sys/ioctl.h>
#define nchar 20 
#define Nbits 16
#define RNG 5.0 //set for +/- 5 Volts (Arduino RANGE5V)
//#define RNG 10.0 //set for +/- 10 Volts (Arduino RANGE10V)
#define BLOCKLEN 1900 // if hangs at bytes_avail, then decrease BLOCKLEN



//Function Prototypes
	int   setprop(int serialfd, struct termios attribs, short debug);
	int   CheckDTnchan(int ndt, int nchannels);
	int   getparm(int argc,char *argv[],
	        int *nsamp, int *ndt, int *nchannels, int *nstack,
		int *fnum, char *source, char *phones, short *nSP,float *sx,
		float rx[]);
	float Time(int *sendcount, int nchannels,int ndt, int *tmod,
		int *tcount);
	int  saveplot (int y[], int npts, int nchannels, int ndt, int chnl, int BIAS );
	int  meanval(int y[],int npts, int sendcount, int nchannels, int chnl, int BIAS) ;
	void writestring(char STRING[], char RES, char LTC,FILE *h1);
	int  WriteSEG2(int npts, int nchnl, float fsamin, int s1[],
	     int ybase[], char ofile[], int stack, char source[],char phones[],
	     short nSP, float sx, float rx[]); 

int main(int argc, char *argv[]) {
    struct termios attribs;
    short serialfd;
    
    char snd[nchar+1];
    char c;
    int  i;
    unsigned short  ivalue= 0;
    short ivalueLow;
    short ivalueHigh;
    int  lvalue = 0;
    int nsamp;
    int ndt;
    int icount=0;
    int istop=0;
    int nstack = 1;
    int nchannels = 1;
    char fname[10];
    unsigned char Lbyte;
    unsigned char Hbyte;
    short debug = 0;
    int fnum = 0;
    int chnl; //channel number to plot
    float fsamin;
    short nSP;
    float sx;
    float rx[8];
    
//    clock_t start,stop;
//    float deltexe;
    char deviceserial[] = "/dev/ttyACM0";
    //const int imax = floor(pow((double)2.,(double)(Nbits-1)));
    const unsigned int imax = floor(pow((double)2.,(double)(Nbits)));
    const float diff = RNG/(float)imax;  //volts/count
    float mvolts;
    int    microvolts;
    int   mspace;
    int  *s1;
int bytes_avail;
int sendcount = -1;
int block = 0;
int left = 0;
int nmax  ;
short iard;
int ybase[8];
char source[2];
char phones[10];

/* Alternative Recording of Data with or without DC level
Set BIAS to virtual ground value in Microvolts above chassis ground */

int BIAS = 0;  //if BIAS == 0 then compute mean and subtract, no DC saved
//int BIAS = 1949000; // BIAS used instead of mean value, DC saved 5Volt case
//int BIAS = 3874000; // BIAS used instead of mean value, DC saved 9Volt case


	if (access("/dev/ttyACM0",F_OK) != -1 ) {
	sprintf(deviceserial,"/dev/ttyACM0");
	//fprintf(stdout,"Serial Device = /dev/ttyACM0 \n"); 
	}

	if (access("/dev/ttyACM1",F_OK) != -1 ) {
	sprintf(deviceserial,"/dev/ttyACM1");
	//fprintf(stdout,"Serial Device = /dev/ttyACM1 \n"); 
	}

// open serial port
	serialfd = open(deviceserial,O_RDWR | O_NOCTTY | O_NDELAY);
	if (serialfd == -1) {
	fprintf(stderr,"ABORT failed to open port \n");
	} //endif


// set serial properties
       if (setprop(serialfd, attribs, debug) != 0) {
	perror(" setprop failure"); }

//start loop
	while (istop != 1) {


// Send number of samples to arduino
//	fprintf(stdout,"Clocks per second = %ld \n",CLOCKS_PER_SEC);
     if (icount == 0) {

// blank snd array
	for (i=0; i<nchar; i++) {
	snd[i]=' ';
	}
	snd[nchar]='\0';

	if(getparm(argc,argv,&nsamp,&ndt,&nchannels,&nstack,&fnum,
	source,phones,&nSP,&sx,rx) !=0) return(1);

 	nmax = nsamp*nchannels ;

// ndt =sample interval in microseconds
	fsamin = ((float) ndt)/1.E6;
        sprintf(fname,"%4.4d.DAT",fnum);

// Allocate RAM for recording 
	mspace = nsamp*nchannels;
	s1 = (int*) calloc(mspace,sizeof(int));
	          if(s1==NULL)
         {
          fprintf(stderr,"ABORT:s1 can't allocate enough RAM \n");
          fprintf(stderr," npts = %d",mspace);
          return(1);
         }

// Zero out s1
	for (i=0; i<mspace; i++) s1[i]=0;

// Check overhead for specified sample interval	
	ndt = CheckDTnchan(ndt,nchannels);
	if (ndt == 0) {	
	puts("ABORT !! sample interval channel error");
	return (1);}

        if(debug !=0 ) fprintf(stderr,"ndt =%d  nchannels=%d \n",ndt,nchannels);

	sprintf(snd,"%5.5d %5.5d %5.5d",nsamp,ndt,nchannels);
	snd[nchar-1]='\n';
	snd[nchar]='\0'; }  
	else {
	sprintf(snd,"%5.5d %5.5d %5.5d",nsamp,ndt,nchannels);
	snd[nchar-1]='\n';
	snd[nchar]='\0'; 
//	puts("ready for trigger ");
     } //endif

//        fprintf(stdout,"snd=:%s \n",snd);

//send paramaters to Arduino
	if(write(serialfd,&snd,nchar) < 0) {
	perror("failed to write");
	}
	if(debug !=0) fprintf(stdout,"snd = %s \n",snd);
//---------------------------test code------------------------

	iard =0;
	while (iard ==0) {
	ioctl(serialfd,FIONREAD,&bytes_avail);
	if (bytes_avail >= 0) {
         read(serialfd,&c,1);
		if (c == 'H' ) {
	  	//fprintf(stdout,"%2.2d Arduino Ready:  ",icount);
	  	fprintf(stdout,"Stack %2.2d: ",icount+1);
		iard=1;
		}
	}
	} //endwhile

//---------------------------test code------------------------


//	puts("ready for trigger ");

i=0;
Hbyte=0;
Lbyte=0;
c='N';
 sendcount = -1;
 block = 0;
 left = 0;
 nmax = nsamp*nchannels ;
// check if more than buffer bytes to send
	if (nmax > BLOCKLEN) {
	block = BLOCKLEN;
	left = nmax - BLOCKLEN;}
	else {
	block =nmax;
	left = 0; }

  while(c !='q')   // Receive from Arduino until 'q' received
  {


	ioctl(serialfd,FIONREAD,&bytes_avail);
//	fprintf(stderr,"available =%d \n",bytes_avail);

         if (bytes_avail >= 2*block )   // check how many bytes available
        {
	for (i=0; i<block; i++) {
         read(serialfd,&Lbyte,1);
	 ivalue=Lbyte;
	 read(serialfd,&Hbyte,1);
	 ivalue=Hbyte*256+ivalue; 
         mvolts = ((float)(ivalue) * diff)*1000.0;
	 microvolts = floor( (mvolts * 1000.0 +.5)) ;

//**    stime=Time(&sendcount,nchannels,ndt,&tmod,&tcount);
	sendcount = sendcount + 1;
//      stime=((float) (tcount-1) * ndt)/1E6; // time (seconds)
//      fprintf(h1," %f %f \n",stime,mvolts);
//**    fprintf(h1," %f %8.8ld \n",stimfsamine,microvolts);
//      fprintf(h1," %f %d \n",stime,ivalue);
	s1[sendcount] = microvolts + s1[sendcount];

	} // next i 

//      fprintf(stderr,"left = %d",left);
	if (left == 0) {
	c = 'q'; } //endif

	if (left >BLOCKLEN) {
	left = left - BLOCKLEN; 
	block = BLOCKLEN;} //do another block of samples
	else {
	block = left;
	left = 0;} //endif
	
        } //endif bytes_avail
 }  //end while c !='q'
	if(debug != 0) fprintf(stderr,"finished q \n"); 

// Get duration in msec
	bytes_avail=0;
        while (bytes_avail < 4) {
	ioctl(serialfd,FIONREAD,&bytes_avail);
	}
//         if (bytes_avail == 4) {
	if(debug != 0) fprintf(stderr,"bytes avail=%d \n",bytes_avail);
// Lower 16 bits 
         read(serialfd,&Lbyte,1);
	 ivalue=Lbyte;
	 read(serialfd,&Hbyte,1);
	 ivalueLow=Hbyte*256+ivalue;
// Higher 16 bits 
         read(serialfd,&Lbyte,1);
	 ivalue=Lbyte;
	 read(serialfd,&Hbyte,1);
	 ivalueHigh=Hbyte*256+ivalue;

	 lvalue = ivalueHigh << 16;
	 lvalue =ivalueLow + lvalue;


	fprintf(stdout,"Record Length %d (msec) \n",lvalue);
//	} //endif 4 bytes



   if(debug !=0 ) fprintf(stdout," %d Samples Read by SEIS16v2 \n",sendcount);
   icount += 1;
   if(icount > nstack-1 ) istop =1;


 } //end while istop !=1

// Open Output File
	icount = 0;
/*
        sprintf(fname,"%8.8d.dat",fnum);
	FILE *h1;
	if ( (h1=fopen(fname,"w"))==NULL) 
        {
          printf("\n ABORT !! Error opening output.dat");
          exit(1);
        }

// write from RAM to a file
	int stime;
	int tmod;
	int tcount;
	sendcount=-1;
	for (i = 0; i < (nmax); i++)
	{ 
	stime=Time(&sendcount,nchannels,ndt,&tmod,&tcount);
        fprintf(h1," %f %8.8d \n",stime,s1[i]);
	} //next sendcount
   fclose(h1);
*/
// find mean value each channel
	for (chnl =0; chnl < nchannels; chnl++) {	
	sendcount = -1;
	ybase[chnl] = meanval(s1,nmax,sendcount,nchannels,chnl,BIAS);
//fprintf(stderr,"meanval = %d \n",ybase[chnl]);
	}


// write from RAM to a file
/*NOTE:  WriteSEG2 removes mean from data so SEG-2 file will have traces with
	zero mean. If run before saveplot(), the plotx.png files will also be
	without mean. If you want to see the mean, move saveplot() loop (below) above 
	WriteSEG2() call.  */
	WriteSEG2(nsamp,nchannels,fsamin,s1,ybase,fname,nstack,source,phones,nSP,
	sx,rx); 

// write plot files (channel numbers 0 to 7).  Move above WriteSEG2 to show mean
	for (chnl = 0; chnl < nchannels; chnl ++) {
	saveplot (s1,nmax,nchannels,ndt,chnl,BIAS);
	} //next chnl

//   read(serialfd,&c,1);
   close(serialfd);
   free(s1);
    return 0;
} //end main


float	Time(int *sendcount, int nchannels, int ndt, int *tmod, int *tcount)
        {
	 float stime;	 
	 *sendcount = *sendcount + 1;
	 *tmod = *sendcount % nchannels;
	 if (*tmod == 0)
		{*tcount=*tcount + 1 ;} 
         stime=((float) (*tcount-1) * ndt)/1E6; // time (seconds)
	 return(stime);
	}  //end Time



//------------SETPROP FUNCTION------------------------------------
int setprop(int serialfd, struct termios attribs, short debug)
{
	speed_t speed;


/* Check that serialfd points to a TTY device */
	if(!isatty(serialfd)) {
	fprintf(stderr,"ABORT, not TTY\n");
	return(1); } //endif

/* get the current settings */
	if(tcgetattr(serialfd, &attribs) < 0) {
	fprintf(stderr,"ABORT, can not get configuration\n");
	return(1);} //endif

if(debug !=0 ) {
// debug print structure
	fprintf(stderr,"GET TERMIOS STRUCTURE\n");
	fprintf(stderr,"c_iflag: %08x \n",attribs.c_iflag);
	fprintf(stderr,"c_oflag: %08x \n",attribs.c_oflag);
	fprintf(stderr,"c_cflag: %08x \n",attribs.c_cflag);
	fprintf(stderr,"c_lflag: %08x \n",attribs.c_lflag);
	fprintf(stderr,"c_cc[NCCS]: %08x \n",attribs.c_cc[NCCS]);
	} //endif


// turn off input processing	
	attribs.c_iflag &= ~(IGNBRK | BRKINT | ICRNL |
	INLCR | PARMRK | INPCK | ISTRIP | IXON);

// No line processing
	attribs.c_lflag &= ~(ECHO | ECHONL | ICANON | IEXTEN | ISIG);

// Turn off character processing
	attribs.c_cflag &= ~(CSIZE | PARENB);
	attribs.c_cflag |= CS8;

// One byte is enought to return from read
	attribs.c_cc[VMIN] = 1;
	attribs.c_cc[VTIME] = 0;

/* set the baudrate */
cfsetospeed(&attribs, B115200); /* outut baudrate */
cfsetispeed(&attribs, B115200); /* input baudrate */
//cfsetospeed(&attribs, 4096); /* outut baudrate */
//cfsetispeed(&attribs, 4096); /* input baudrate */

if(debug !=0 ) {
// debug print structure
	fprintf(stderr,"\nSETTING TERMIOS STRUCTURE\n");
	fprintf(stderr,"c_iflag: %08x \n",attribs.c_iflag);
	fprintf(stderr,"c_oflag: %08x \n",attribs.c_oflag);
	fprintf(stderr,"c_cflag: %08x \n",attribs.c_cflag);
	fprintf(stderr,"c_lflag: %08x \n",attribs.c_lflag);
	fprintf(stderr,"c_cc[NCCS]: %08x \n",attribs.c_cc[NCCS]);
	} //endif


/* if there is need for it, set other settings here */

/* eventually apply everything for your serialfd descriptor */
//if (tcsetattr(serialfd, TCSANOW, &attribs) < 0)
if (tcsetattr(serialfd, TCSANOW, &attribs) < 0)
{
        perror("stdin");
        return(1) ;
    }

if(debug !=0 ) {
	speed = cfgetispeed(&attribs);
	fprintf(stderr,"input speed: %lu\n", (unsigned long) speed);
	speed = cfgetospeed(&attribs);
	fprintf(stderr,"output speed: %lu\n", (unsigned long) speed);
	puts("---------------\n");
	} //endif


  return(0);
}  //end setprop
//----------------------------------------------------------------

/* Function to check ndt > overhead, optimization 3 on arduino */
	int  CheckDTnchan(int ndt, int nchannels)
	{
	switch (nchannels) {
		case 1 :
		if (ndt < 1000) return(1000); else return(ndt);
		break;
		case 2 :
		if (ndt < 2000) return(2000); else return(ndt);
		break;
		case 3 :
		if (ndt < 2000) return(2000); else return(ndt);
		break;
		case 4 :
		if (ndt < 3000) return(3000); else return(ndt);
		break;
		case 5 :
		if (ndt < 3000) return(3000); else return(ndt);
		break;
		case 6 :
		if (ndt < 4000) return(4000); else return(ndt);
		break;
		case 7 :
		if (ndt < 4000) return(4000); else return(ndt);
		break;
		case 8 :
		if (ndt < 5000) return(5000); else return(ndt);
		default :
		return (0);
	} //endswitch

	} //end CheckDTnchan

	int getparm(int argc, char *argv[],
	    int *nsamp, int *ndt, int *nchannels, int *nstack, int *fnum,
		char source[], char phones[], short *nSP, float *sx,
		float rx[])
	{
	int i;

	if (argc == 1 ) {
	fprintf(stderr,"USAGE: %s nsamp ndt nchannels nstack file source phones SP SX RX \n",
	argv[0]);
	puts("Ex: SEIS16v2 1000 5000 8 1 0 H VRTVRTVV 0 5 1 2 3 4 5 6 7 8");
	puts("source: H=hammer W=weight_drop D=dynamite G=gun V=vibrator");
	puts("phones: V=vertical R=radial T=transverse (8 channels)");
	puts("SP: Shot point number, SX Shot point X-coord");
	puts("RX: Array of X-coordinates geophones");
	return(1); }
	else {
	if (argc >=2) *nsamp = atoi(argv[1]); else *nsamp = 1000;
	if (argc >=3) *ndt = atoi(argv[2]); else *ndt = 1000;
	if (argc >=4) *nchannels = atoi(argv[3]); else *nchannels = 1;
	if (argc >=5) *nstack = atoi(argv[4]); else *nstack =1;
	if (argc >=6 ) *fnum = atoi(argv[5]); else *fnum = 0;
	if (argc >= 7) strncpy(source,argv[6],1);
	if (argc >= 8) strncpy(phones,argv[7],8);
	if (argc >=9) *nSP = atoi(argv[8]); else *nSP = 0;
	if (argc >=10 ) *sx = atof(argv[9]); else *sx = 0;
		for ( i = 11; i<19; i++) {
		if (argc >=i ) rx[i-11] = atof(argv[i-1]); else rx[i-11] = 0;
		} //next i	
	} //end else
	return(0);	
	}  //end getparm

// SAVE PLOT-----------------------------------------------------------------
	int  saveplot (int y[], int npts, int nchannels, int ndt, int chnl, int BIAS )
	{
	//y[] is in microvolts
	int tmod;
	int tcount;
	int sendcount;
	float stime;
	int itrace;
	char	XL[9]="Time (s)";
	char    YL[23]="Amplitude (millivolts)";
	char Xwin[4]="png";
	char ofile[10];
	int   ybase =  0;
	int   dscale = 1;
	if (BIAS != 0 ) {dscale =1000;}
	sendcount = -1;
	ybase = meanval(y,npts,sendcount,nchannels,chnl,BIAS)/1000; // mv
	sprintf(ofile,"plot%1.1d.png",chnl);
	itrace = chnl;
	int FG; //fixed gain variable
	int DSFI; //descale factor inverse
/* Hardwired Gain is 19dbv (the 1000 microvolts to mv in gnuplot pipe) */
	FG = 19;
   	DSFI = round ( (pow((double) 10., (double) ((float) FG)/20.0)));


/*
	FILE *h2; 
//...open output file
        if( (h2=fopen(ofile,"w"))==NULL ) {
          fprintf(stderr,"\n ABORT !! Error opening file %s",ofile);
          return(1); }
*/

//  PIPE VERSION
	FILE *h2 =popen("gnuplot ","w");

  	fprintf(h2,"set grid \n");
  	fprintf(h2,"set output \"%s\" \n",ofile);
  	fprintf(h2,"set terminal %s size 640,240  \n",Xwin);
  	fprintf(h2,"set key left \n");
  	fprintf(h2,"set xlabel \" %s \" \n",XL);
  	fprintf(h2,"set ylabel \" %s \" \n",YL);
//	fprintf(h2,"set yrange [-500:500] \n");
	//note plot in mv ($2/1000) and data writen below corrected for k-gain DSFI
  	fprintf(h2," plot '-' u ($1):($2/1000):($3) lc 'black' w filledc above t '' ,");
	fprintf(h2," '' u ($1):($2/1000) lc 'black' w l t \"  Trace %d \" \n",itrace+1);
  	int j;
		tmod =0;
		tcount = 0;
		sendcount = -1;
  		for(j=0;j<npts;j++) {
		stime=Time(&sendcount,nchannels,ndt,&tmod,&tcount);
			//Select chnl
			tmod = sendcount % nchannels;
			if (tmod == chnl) {	
    			fprintf(h2,"%f %d %d \n",stime,y[j]/DSFI,ybase/dscale);
			} //endif
  		} //next j 
  	fprintf(h2,"e\n");  //this is required when std in - plot
		tmod =0;
		tcount = 0;
		sendcount = -1;
 		for(j=0;j<npts;j++) {
		stime=Time(&sendcount,nchannels,ndt,&tmod,&tcount);
			//Select chnl
			tmod = sendcount % nchannels;
			if (tmod == chnl) {	
    			fprintf(h2,"%f %d %d \n",stime,y[j]/DSFI,ybase/dscale);
			} //endif
  		} //next j 
  	fprintf(h2,"e\n");  //this is required when std in - plot
	fclose(h2);
	return(0);
//...close pipe
	fflush(h2);
	pclose(h2);
	} //end saveplot

//COMPUTE MEAN============
	int meanval(int y[],int nmax, int sendcount,int nchannels,int chnl, int BIAS) {
	int i;
	long sum = 0;  //even though a sample fits, 1000's of samples gets big
	int average;
	int tmod;
	if (BIAS == 0 ) {
	for (i=0; i<nmax; i++) {
		sendcount = sendcount +1;
		tmod = sendcount % nchannels;
		if (tmod == chnl) {	
		sum = sum + y[i];
		} //endif
	} //next i
	average = sum / (nmax/nchannels);
	} else {
	average = BIAS; }
	return(average);
	} //end meanval

//=========================WriteSEG2=============================================	
int	WriteSEG2(int npts, int nchnl, float fsamin, int s1[],
	 int ybase[],char rawrecord[], int stack, char source[], char phones[],
	short nSP, float sx, float rx[]) 
{
//...declare variables
	time_t            start;
	struct tm        *ptr;
	char TIME[40], DATE[40];
//C++ requires char const, not just char
	char const *mon[12]={"Jan","Feb","Mar","Apr","May","Jun",
        "Jul","Aug","Sep","Oct","Nov","Dec"};

	FILE    *h1;
	char ofile[12] ;
	int	i, k;
	unsigned short int ID = 0x3a55;  				//bytes 0 1
	unsigned short int REV = 1;					//bytes 2 3
	/* Note: 8 channel recorder, but allow max 10 channels for pointers space
	This will permit some zeros for buffer to free format section */
	unsigned short int M ;  // size of trace point block NCH*4  //bytes 4 5
	unsigned short int NCH   ; // number of channels 		//bytes 6 7
	char LENST = 0x01; // length of string terminator  (bytes)	//byte  8
	char ST = 0; //string terminator byte				//bytes 9 10
	char LT = 0x01; //length of line terminator character (bytes)	//byte  11
	char LTC = 0x0a; // line terminator				//byte  12
	char RES = 0x00; //reserved byte				//bytes 14-31
	int ZERO4 = 0;   //4 byte zero
	short int ZERO2 = 0;   //2 byte zero
	unsigned short int TDBID = 0x4422;
	unsigned long int TPB, LOC1, LOC2, EFDB, EDB;
	int CHNL ;
//	int stack ;
	char STRING[40];
	int FG; //fixed gain variable
	double DSF; //descale factor


//...set parameters
	char lineid[] = "0001";
	sprintf(ofile,"%s",rawrecord);
	float	ry = 0.0,
		rz = 0.0;
	float 	sy = 0.0,
		sz = 0.0;
//	stack = 1;
//	nchnl = 8;

	char typerecvr[30];
	char typesrc[30];
//	char source = 'H';
	switch (source[0]) {
		case 'H':
		sprintf(typesrc,"HAMMER");
		break;
		case 'W':
		sprintf(typesrc,"WEIGHT_DROP");
		break;	
		case 'D':
		sprintf(typesrc,"DYNAMITE");
		break;	
		case 'G':
		sprintf(typesrc,"GUN");
		break;	
		case 'V':
		sprintf(typesrc,"VIBRATOR");
		break;	
		default:
		sprintf(typesrc,"HAMMER");
		}

	NCH = nchnl; 
	CHNL = 0;
	M=32;  // NCH * 4 if NCH max = 8

	for (i=0; i<40; i++) {
	TIME[i] = '\0';
	DATE[i] = '\0'; }


//...open output file
        if( (h1=fopen(ofile,"wb"))==NULL )
	{
	fprintf(stderr,"\n ABORT !! Error opening file %s",ofile);
	return(1);
	}

//...GET TIME AND DATE
      time(&start);
      ptr=localtime(&start);
      sprintf(TIME,"  ACQUISITION_TIME %2.2u:%2.2u:%2.2u  ",
         (*ptr).tm_hour,(*ptr).tm_min,(*ptr).tm_sec);
      sprintf(DATE,"  ACQUISITION_DATE %2.2d/%3s/%4.4d  ",
         (*ptr).tm_mday,mon[(*ptr).tm_mon],(*ptr).tm_year+1900);
	
//...FILE DESCRIPTOR BLOCK
	fwrite(&ID,sizeof(short int),1,h1); //bytes 0,1 ID
	fwrite(&REV,sizeof(short int),1,h1); //bytes 2,3 revision number
	fwrite(&M,sizeof(short int),1,h1);//bytes 4,5 size trace pointer block
	fwrite(&NCH,sizeof(short int),1,h1); //bytes 6,7 number of channels
	fwrite(&LENST,sizeof(char),1,h1);  // byte 8 size of ST
	fwrite(&ST,sizeof(char),1,h1); //byte 9   string term char used
	fwrite(&ST,sizeof(char),1,h1); //byte 10  2nd string term char
	fwrite(&LT,sizeof(char),1,h1); //byte 11  size LTC
	fwrite(&LTC,sizeof(char),1,h1); //byte 12 line term char
	fwrite(&ST,sizeof(char),1,h1); //byte 13 2nd line term char not used
	for (i=14; i<32; i++) {
	fwrite(&RES,sizeof(char),1,h1); }

//...TRACEPOINTER SUBBLOCK
//	TPB = ftell(h1);    //capture start of block to come back to later
	TPB = (long int) 0x20;
	for (i=0; i<8; i++) {
	fwrite(&ZERO4,sizeof(int),1,h1);} //zero out pointers to start 

//...Write Date of acquisition
	LOC1=ftell(h1);
	writestring(DATE,RES,LTC,h1);

//...Write Time of acquisition
	writestring(TIME,RES,LTC,h1);

//...Write Instrument 
	sprintf(STRING,"  INSTRUMENT PM_Arduino Mayhew_A/D 01  ");
	writestring(STRING,RES,LTC,h1);

//...Write Instrument 
	char UNITS[40];
	sprintf(UNITS,"  UNITS METERS  ");
	writestring(UNITS,RES,LTC,h1);

//...capture end of free format location
	EFDB=ftell(h1) - 1 ;
//	printf("location end of FDB = %0lx \n",EFDB);

	EDB = EFDB ;  //not really end of data block first time
			// EDB is also start of TDB later on
	
/* START OF TRACE DESCRIPTOR BLOCK ===================================== */

	for (k=0; k<nchnl; k++) {

//return to trace pointer block start and write LOC2
	fseek(h1,TPB+k*4,SEEK_SET);
	LOC2= EDB + 1 ;  // add one to point beyond end of FDB or EDB
	fwrite(&LOC2,sizeof(int),1,h1);


//...Write Trace Descriptor block
	fseek(h1,LOC2,SEEK_SET);  //return to start of trace descriptor block
//printf("%08lx \n",LOC2);
	fwrite(&TDBID,sizeof(short int),1,h1);  //write ID code for TDB

	LOC1=ftell(h1);  //start of trace descriptor block just after ID
	short int TDBS = ZERO2;
	fwrite(&TDBS,sizeof(short int),1,h1); //write zero temporary for block size
	int SB = 4*npts;	//size of data block in bytes
	fwrite(&SB,sizeof(int),1,h1); //size of data block in bytes
	int NS = npts; 		// number of samples in a data block
	fwrite(&NS,sizeof(int),1,h1); //number of samples in data block
	char DFC = 0x02; // Data Format Code for 32 bit integer		
	fwrite(&DFC,sizeof(char),1,h1);
	/* Reserved */

/* NOTE Strings must start with 2 spaces to allow offset to next replacement int
        and must end with 2 spaces to allow for \0 and \0a replacement (eofs eofl) */
	for (i=13; i<32; i++) {
	fwrite(&RES,sizeof(char),1,h1);
	 }
	sprintf(STRING ,"  CHANNEL_NUMBER %d  ",CHNL+k);
	writestring(STRING ,RES,LTC,h1);

	sprintf(STRING,"  ALIAS_FILTER %d %d  ",100,6);
	writestring(STRING,RES,LTC,h1);

	sprintf(STRING,"  HIGH_CUT_FILTER %d %d  ",100,6);
	writestring(STRING,RES,LTC,h1);

	FG = 19;  //gain in dBv (change if your instrument different)
	sprintf(STRING,"  FIXED_GAIN %d  ",FG);
	writestring(STRING,RES,LTC,h1);

	sprintf(STRING,"  DELAY %.2f  ",0.);
	writestring(STRING,RES,LTC,h1);

	sprintf(STRING,"  LINE_ID %s  ",lineid);
	writestring(STRING,RES,LTC,h1);

	sprintf(STRING,"  RAW_RECORD %s  ",rawrecord);
	writestring(STRING,RES,LTC,h1);

	switch (phones[k]) {
		case 'V':
		sprintf(typerecvr,"VERTICAL_GEOPHONE");
		break;
		case 'T':
		sprintf(typerecvr,"T_HORIZONTAL_GEOPHONE");
		break;
		case 'R':
		sprintf(typerecvr,"R_HORIZONTAL_GEOPHONE");
		break;
		default:
		sprintf(typerecvr,"VERTICAL_GEOPHONE");
		}

	sprintf(STRING,"  RECEIVER %s  ",typerecvr);
	writestring(STRING,RES,LTC,h1);

	sprintf(STRING,"  SOURCE %s  ",typesrc);
	writestring(STRING,RES,LTC,h1);

	sprintf(STRING,"  SOURCE_STATION_NUMBER %4.4d  ",nSP);
	writestring(STRING,RES,LTC,h1);

	sprintf(STRING,"  RECEIVER_LOCATION %.2f %.2f %.2f  ",rx[k],ry,rz);
	writestring(STRING,RES,LTC,h1);

	sprintf(STRING,"  SOURCE_LOCATION %.2f %.2f %.2f  ",sx,sy,sz);
	writestring(STRING,RES,LTC,h1);

	sprintf(STRING,"  SAMPLE_INTERVAL %.6f  ",fsamin);
	/* Arduino A/D -->SEIS16v2 s1[] is in microvolts, SEG-2 Standard 
	requires a descaling factor applied to the SEG-2 data which will
	yield mvolts */
	writestring(STRING,RES,LTC,h1); 
	
   	DSF = 1.000 / (1000. * pow((double) 10., (double) ((float) FG)/20.0));
	sprintf(STRING,"  DESCALING_FACTOR %14.7E  ",DSF);
	writestring(STRING,RES,LTC,h1);

	sprintf(STRING,"  STACK %d  ",stack);
	writestring(STRING,RES,LTC,h1);

	LOC2 = ftell(h1);
	TDBS = LOC2 - LOC1 + 2 ;  // add 2 bytes to include ID 0x4422

//determine padding if needed for Trace Descriptor Block TDB
//...Size, TDBS must be divisible by 4
	int pad,p1;
	p1=TDBS % 4;
	if ( p1 != 0) {
		pad = 4-p1;
		TDBS = TDBS + pad;
	} //endif
//printf("pad = %d \n",pad);
//	printf("Trace Descriptor Block Size (bytes) %d \n",TDBS);	
	fseek(h1,LOC1,SEEK_SET);
	fwrite(&TDBS,sizeof(short int),1,h1); //write block size
	fseek(h1,LOC2,SEEK_SET);

	if ( p1 != 0) {
		for (i=0; i<pad; i++) fwrite(&ST,sizeof(char),1,h1);
	}
//	long LOC3;
//	LOC3=ftell(h1);
//	printf("LOC3 = %8lx \n",LOC3);

//...WRITE DATA
	int *pr;
	pr = &s1[0] + k;
	/*NOTE: DC level is removed from data since pr changes signal
	 on exit:  *pr = *pr - ybase[k] changes s1, a pointer */
	for (i=0; i<npts; i=i+1) {
	*pr = *pr - ybase[k];
	fwrite(pr,sizeof(int),1,h1);
	pr = pr + nchnl;
	} //end of data write
	
//...save end of data block
	EDB = ftell(h1) - 1; //the last byte of 4byte integer

	} // next k chanl
	fclose(h1);	
	return(0);
} //end WriteSEG2

void writestring(char STRING[], char RES, char LTC,FILE *h1)
{
	int nbytes, nbytesf;
	nbytes = fprintf(h1,"%s",STRING);
	fseek(h1,-nbytes,SEEK_CUR);
	nbytesf = nbytes + 1  ;  //replace 2 blanks at begin with stringlength
//		printf("bytes %d \n",(int) nbytes);
	fwrite(&nbytesf,sizeof(short int),1,h1); //write number of bytes forward
						//to next 2byte forward of next
	fseek(h1,nbytes-3,SEEK_CUR);
	fwrite(&RES,sizeof(char),1,h1); //replace 2nd last char with \0 endofstring 
	fwrite(&LTC,sizeof(char),1,h1); //replace last char with 0xa endofstring 
}
