%   |------------------------------------------------------------|
%   | $Id: FwdR1.m,v 1.1.1.1 2019/04/29 22:51:20 pm Exp $        |
%   | Basic Seismic Utilities (BSU)                              |
%   | LANGUAGE: octave                                           |
%   | AUTHOR: P. MICHAELS                                        |
%   | FUNCTION NAME: FwdR1.m                                     |
%   | DIRECTORY: BSUROOT/octave                                  |
%   |  revised for OCTAVE                                        |
%   |------------------------------------------------------------|
%   |Fowrward problem, Rayleigh wave given bvax results          |
%   |to compare to.  model.txt is a file with 3 rows             |
%   |Example: for nlay=3 vi=shear velocity, zi=depth layer top   |
%   | nlay                                                       |
%   | v1 v2 v3                                                   |
%   | z1 z2 z3                                                   |
%   |------------------------------------------------------------|
%   | octave is an interpreter useful for scientifc calculations.|
%   | It is similar to MATLAB and may be obtained free from      |
%   | http://www.gnu.org/software/octave                         |
%   |------------------------------------------------------------|
%   |  Note: requires disper.oct (run build_disper_oct to build) |
%   |------------------------------------------------------------|
%   |  NOTICE:                                                   |
%   |  Copyright (c) 2017 Paul Michaels                          |
%   |  <pm@cgiss.boisestate.edu>                                 |
%   |  This program is free software; you can                    |
%   |  redistribute it and/or modify it under the terms          |
%   |  of the GNU General Public License as published            |
%   |  by the Free Software Foundation; either version           |
%   |  3 of the License, or (at your option) any later           |
%   |  version.  This program is distributed in the              |
%   |  hope that it will be useful, but                          |
%   |  WITHOUT ANY WARRANTY; without even the implied            |
%   |  warranty of MERCHANTABILITY or FITNESS FOR A              |
%   |  PARTICULAR PURPOSE.  See the GNU General Public           |
%   |  License for more details.                                 |
%   |  You should have received a copy of the GNU                |
%   |  General Public License along with this program;           |
%   |  if not, write to the Free Software Foundation,            |
%   |  Inc., 675 Mass Ave, Cambridge, MA 02139, USA.             |
%   |------------------------------------------------------------|

clear
function [grho,gmu,glame]=gparms(rho,mu,lame,zi,nlay,depth)
      l1=1;
      for j=1:nlay
      	if(depth>zi(j)); 
      	l1=j ;
	end
      end
      if(l1<nlay) 
      l2=l1+1;
      dz=zi(l2) - zi(l1);
      sloper=(rho(l2)-rho(l1))/dz;
      slopel=(lame(l2)-lame(l1))/dz;
      slopem=(mu(l2)-mu(l1))/dz;
      z=depth-zi(l1);
      grho=sloper*z + rho(l1);
      gmu=slopem*z + mu(l1);
      glame=slopel*z + lame(l1);
      else
      grho=rho(nlay);
      gmu=mu(nlay);
      glame=lame(nlay);
      end
end

function [alpha,beta1]=e2v(Rho,Mu,Lame)
beta1=sqrt(Mu./Rho);
alpha=sqrt(Lame./Rho+2*beta1^2);
end

function [c]=dcal(Nfreq,nlay,zi,Alpha,Beta,rho,deltz,Freq)
%compute calculated data
for j=1:length(zi)
mu(j,1)=rho(j)*Beta(j)^2; % shear modulus
lame(j,1)=rho(j)*(Alpha(j)^2-2.00*Beta(j)^2); % Lame's constant
end
for j=1:Nfreq
frq1=Freq(j);
maxmod=1;
cvel=zeros(maxmod,1);
cvel=disper(nlay,rho,mu,lame,zi,deltz,frq1,maxmod,cvel);
c(j,1)=cvel(1);
end
end

function [A,B,Z,R]=plotvel(iwin,clrsw,zi,deltz,rho,mu,lame,nlay)
%plot velocity model
nz=floor(zi(nlay)/deltz+.5);
zo=zi(1);
for j=1:2:nz*2
z1=zo+deltz;
depth=(z1+zo)/2;
[grho,gmu,glame]=gparms(rho,mu,lame,zi,nlay,depth);
[alpha,beta1]=e2v(grho,gmu,glame);
A(j,1)=alpha;
B(j,1)=beta1;
Z(j,1)=zo;
R(j,1)=grho;
A(j+1,1)=alpha;
B(j+1,1)=beta1;
Z(j+1,1)=z1;
R(j+1,1)=grho;
zo=z1;
end %next j
figure(iwin)
clf
hold on
if clrsw==0 
plot(A,-Z,'r') %p-wave velocity
plot(B,-Z,'b') %s-wave velocity
plot(R,-Z) %density
else
plot(A,-Z) %p-wave velocity
plot(B,-Z) %s-wave velocity
end
grid;
end


function [VSave]=vsave(zi,deltz,rho,mu,lame,nlay)
%VSave is Vs30 when max depth of model is 30 meters
nz=floor(zi(nlay)/deltz+.5);
zo=zi(1);
tsum=0;
zmax=zi(nlay);
for j=1:2:nz*2
z1=zo+deltz;
depth=(z1+zo)/2;
[grho,gmu,glame]=gparms(rho,mu,lame,zi,nlay,depth);
[alpha,beta1]=e2v(grho,gmu,glame);
tsum=tsum+deltz/beta1;
zo=z1;
end %next j
VSave=zmax/tsum;
end

function [invar]=editmodel(invar)
N=length(invar)
  for j=1:N
  x=sprintf('%10.4f',invar(j));
  k=length(x);
  a(j,1:k)=x;
  end

b(1,1:6)='Nlayer';
jstop=(N-1)/2+1
  for j=2:jstop
  x=sprintf('Vs(%2.2d)',j-1);
  k=length(x);
  b(j,1:k)=x;
  x=sprintf('Zi(%2.2d)',j-1);
  k=length(x);
  b(jstop+j-1,1:k)=x;
  end 
prompt=cellstr(b)';
defaultans=cellstr(a)';
dlg_title = 'Make Changes';
num_lines = 1;
cn = inputdlg(prompt,dlg_title,num_lines,defaultans);
frect = str2double(cn);
for j=2:N
invar(j)=frect(j);
end
end


%====================begin program====================

igo=1;
while (igo)
c=exist('disper.oct');
 if c == 0
 c2=exist('build_disper_oct');
 c3=exist('wrapper.cpp');
 c4=exist('rwv.f');
   if c2 != 0 && c3 != 0 && c4 !=0
   T=sprintf('Requires disper.oct \nTo run build_disper_oct, CLICK OK');
   msgbox(T);
   system('build_disper_oct');
 else
   T1=sprintf('ABORT\n');
   if c2 ==0 T2=sprintf('build_disper_oct missing\n'); else T2=' '; end
   if c3 ==0 T3=sprintf('wrapper.cpp missing\n'); else T3=' '; end
   if c4 ==0 T4=sprintf('rwv.f missing'); else T4=' '; end 
   T=strcat(T1,T2,T3,T4);
   msgbox(T);
break
end
end

%Initial Model Parameters
%nlay=5;
%Beta=[ 194;104.35; 91.457; 214.78; 350.];  % shear velocities
%zi=[    0; 1.14; 2.320; 12.79; 30.0;]; % depth control points
Query=sprintf('Enter File (nlay;vs;z) \n example: model.txt ');
ifile=input(Query,'s');
invar=ifile(1: findstr(ifile,'.')-1);
fp1=fopen(ifile);
[invar]=fscanf(fp1,'%f');
nlay=invar(1)
Beta(1:nlay)=invar(2:1+nlay)
zi(1:nlay)=invar(2+nlay:1+nlay+nlay)
fclose(fp1)

msg0='Fixed Vp/Vs or Fixed Vp';
BTN1='Vp/Vs ';
BTN2='Vp=fix';
TITLE='Choose Vp';
BTN=questdlg(msg0,TITLE,BTN1,BTN2,BTN1);

if BTN == 'Vp=fix'
  alphafixed=1500;
  rhomoist=2100;
  for j=1:nlay
  Alpha(j,1)=alphafixed;  %alternative to Poisson's Ratio
  rho(j)=rhomoist; %Densities
  end
else
%alpha and rho settings
sigma=1/3;  %Poisson's Ratio to set Alpha
alphafixed=1700; %alternative to sigma
Gs=2.67;    %grain density
npor=.3;    %porosity
evr=npor/(1-npor); %void ratio
Sat=1.0;  %degree water saturation

%get arguments
sigmatxt=sprintf('%.2f',sigma);
Gstxt=sprintf('%.2f',Gs);
nportxt=sprintf('%.2f',npor);
Sattxt=sprintf('%.2f',Sat);
prompt = {'Poisson Ratio','Solid g/cc','porosity','Degree Water Sat.'};
dlg_title = 'Enter Soil Properties for Vp/Vs';
num_lines = 1;
defaultans = {sigmatxt,Gstxt,nportxt,Sattxt};
cn = inputdlg(prompt,dlg_title,num_lines,defaultans);
frect = str2double(cn);
sigma=frect(1); Gs=frect(2); npor=frect(3); Sat=frect(4);

evr=npor/(1-npor); %void ratio
rhomoist=((Gs+Sat*evr)*1000)/(1+evr);  %mass density soil
alphaObeta=sqrt( 2*(1-sigma) / (1-2*sigma));
for j=1:nlay
Alpha(j,1)=Beta(j)*alphaObeta; %P-wave velocities
rho(j)=rhomoist; %Densities
end
end

if BTN == 'Vp=fix'
msg1=sprintf('Vp=%.0f m/s \nDen=%.0f kg/m3',alphafixed,rhomoist);
msgbox(msg1);
else
msg1=sprintf('Vp/Vs = %.2f \nDen=%.0f kg/m3',alphaObeta,rhomoist);
msgbox(msg1);
end

% interval layer thickness and other parameters
deltz=.2; 
maxmod=1;  %fundamental mode only

figure(1)
%observed data
cc=exist('bvax.his');
if cc ~= 0
FILE='bvax.his';
fp1=fopen(FILE,'rt')
data=fscanf(fp1,'%f %f %f %f %f %f \n',[6,Inf]);
fclose(fp1);
Freq=data(1,:)';
dobs=data(2,:)';
em=data(3,:)*1.96'; %95 percent conf
ep=em;
clf
hold off
errorbar(Freq,dobs,em)
grid
hold on
else
Freq=(0.5:0.5:50)';
end

Nfreq=length(Freq);
pertb=.0001;

%start loop==============================================
%itmax=1;
%for iter=1:itmax
cont=1;
iter=0;
while cont == 1
iter=1+iter;


%change parameters

if BTN == 'Vp/Vs '
for j=1:nlay
Alpha(j)=Beta(j)*alphaObeta;
end
end

%compute calculated data
c=dcal(Nfreq,nlay,zi,Alpha,Beta,rho,deltz,Freq);
if cc~=0
e=dobs-c;
esv(iter)=e'*e;
end

figure(1);
clf
hold on
plot(Freq,c,'r');
if cc~=0
errorbar(Freq,dobs,em)
end
%end %next iter
msg0='Continue with new model';
TITLE='Continue';
CBTN=questdlg(msg0,TITLE);
   switch CBTN
     case 'Yes'
     cont=1;
     invar=editmodel(invar);
     Beta(1:nlay)=invar(2:1+nlay)
     zi(1:nlay)=invar(2+nlay:1+nlay+nlay)
     case 'No'
     cont=0;
    end
end %endwhile

T0=sprintf('Rayleigh Wave \n Vs=');
T1=sprintf('%.1f  ',Beta);
T1b=sprintf('\n Zi=');
T2=sprintf('%.3f  ',zi);
if cc ~=0
T3=sprintf('\n deltz=%.3f Lsqe=%.3e',deltz,sqrt(esv(iter)/Nfreq));
else
T3=sprintf('\n deltz=%.3f ',deltz);
end
hold on
plot(Freq,c,'r');
if cc ~= 0
errorbar(Freq,dobs,em,ep)
end
title(strcat(T0,T1,T1b,T2,T3));
xlabel('Frequency Hz');
ylabel('Phase Velocity (m/s)');
grid on

%compute elastic properties
for j=1:length(zi)
mu(j,1)=rho(j)*Beta(j)^2; %shear modulus
lame(j,1)=rho(j)*(Alpha(j)^2-2.00*Beta(j)^2); %Lame's constant
end

[VSave]=vsave(zi,deltz,rho,mu,lame,nlay);
sprintf('Vs30= %.0f m/s',VSave)



%plot velocity and density model
iwin=2;
clrsw=0; %color switch
[A,B,Z,R]=plotvel(iwin,clrsw,zi,deltz,rho,mu,lame,nlay);
hold on
plot(Beta,-zi,'markersize',5,'o');  %plot control points on S-velocity
TM=sprintf('Model: Vs Vp Den  [VSave=%.2f m/s]',VSave);
title(TM)
ylabel('Depth (m)');
xlabel(' Velocity (m/s) or Density (kg/m^3)')

%wavenumber plot
if cc ~=0
wl=dobs./Freq;
end
wlc=c./Freq;
figure(3)
clf
hold on
if cc ~= 0
plot(dobs,-wl,'o','markersize',5)
errorbar(dobs,-wl,em,em,0,0,'~>')
end
plot(c,-wlc,'r')
grid
hold off
xlabel('Velocity (m/s)');
ylabel('Wavelength (m)');
title(strcat(T0,T1,T1b,T2,T3));
igo=0;
end %endwhile

