%   |------------------------------------------------------------|
%   | $Id: KD4kvmb.m,v 1.1.1.1 2019/04/29 22:51:20 pm Exp $      |
%   | Basic Seismic Utilities (BSU)                              |
%   | LANGUAGE: octave                                           |
%   | AUTHOR: P. MICHAELS                                        |
%   | FUNCTION NAME: KD4kvmb.m                                   |
%   | DIRECTORY: BSUROOT/octave                                  |
%   |  revised for OCTAVE                                        |
%   |------------------------------------------------------------|
%   |2DOF System (water and frame)                               |
%   |Requires Porosity, and wave determined Stiffness and Damping|
%   |n = porosity  C1=stiffness m^2/s^2                          |
%   |              C2=damping  m^2/s                             |
%   |SOLUTION:  Solves for hydraulic conductivity (m/s)          |
%   |   NOTE:  There will be up to 2 solutions due to the        |
%   |          convext down nature of the mapping                |
%   |                                                            |
%   |           COUPLED (b case)   Likely solution Kd            |
%   |           UNCOUPLED (a case) Alternative solution Kd       |
%   |           Peak of the curve is also displayed in answer    |
%   |           (no solution if wave damping ratio > curve peak) |
%   |   Based on original Scilab Code                            |
%   |   KD-4X.sci                                                |
%   |                                                            |
%   |------------------------------------------------------------|
%   | octave is an interpreter useful for scientifc calculations.|
%   | It is similar to MATLAB and may be obtained free from      |
%   | http://www.gnu.org/software/octave                         |
%   |------------------------------------------------------------|
%   |  NOTICE:                                                   |
%   |  Copyright (c) 2017 Paul Michaels                          |
%   |  <pm@cgiss.boisestate.edu>                                 |
%   |  This program is free software; you can                    |
%   |  redistribute it and/or modify it under the terms          |
%   |  of the GNU General Public License as published            |
%   |  by the Free Software Foundation; either version           |
%   |  3 of the License, or (at your option) any later           |
%   |  version.  This program is distributed in the              |
%   |  hope that it will be useful, but                          |
%   |  WITHOUT ANY WARRANTY; without even the implied            |
%   |  warranty of MERCHANTABILITY or FITNESS FOR A              |
%   |  PARTICULAR PURPOSE.  See the GNU General Public           |
%   |  License for more details.                                 |
%   |  You should have received a copy of the GNU                |
%   |  General Public License along with this program;           |
%   |  if not, write to the Free Software Foundation,            |
%   |  Inc., 675 Mass Ave, Cambridge, MA 02139, USA.             |
%   |------------------------------------------------------------|

clear all;

%get arguments
prompt = {'Frequency (Hz)','n (porosity)',...
'C1 stiffness (m^2/s^2)','C2 damping (m^2/s)',...
'+/-stdevC1','+/-stdevC2','+/-stdev{porosity'};
dlg_title = 'Enter Parameters';
num_lines = 1;
defaultans = {'12','.25','10000','5','100','1','.01'};
cn = inputdlg(prompt,dlg_title,num_lines,defaultans);
frect = str2double(cn);
freqdr=frect(1); n=frect(2); C1=frect(3); C2=frect(4);
C1E=frect(5)*1.96; C2E=frect(6)*1.96; uPorE=1.96*frect(7);
umeanpor=n;
ustdpor=uPorE;
%----------------------------------------------------------------------
% convert frequency to radians/s, compute L=length resonator
wdr=freqdr*2*pi;      % radian frequency for damping ratio
L=sqrt(C1)/wdr; %length of vibrator
dratioKV=(wdr*C2)/(2*C1);  % damping ratio
% error bar on KV dratio
t1 = C1E^2; 
t2 = wdr^2;
t4 = C2^2;
t5 = C1^2;
t6 = t5^2;
t10 = C2E^2;
sigmadr = sqrt(t1*t2*t4/t6+t10*t2/t5)/2;
%----------------------------------------------------------------------

%------------Compute Permeability-------------------
A=1;  % area of vibrator in meters^2
rhow=1000; %mass density of water
Gs=2.67;   %specific gravity of solids

L=sqrt(C1)/wdr; %length of vibrator
%KVMB Model
global Mf=(1-umeanpor)*Gs*rhow*A*L;  %mass of frame
global Mw=umeanpor*rhow*A*L;         %mass of water
Mkv=Mf+Mw;                %KV mass, coupled
global kf=(C1*Mkv)/(L*L);   %spring
fHZ=sqrt(kf/Mkv)/(pi*2); %frequency implied (Hz)


%KVMB Damping ratio for a given dashpot function
function [delta,frequ]=KVMB(d1)
global Mw;
global Mf;
global kf;
% ODE in matrix form
% d1=dashpot value kg/s
%x-frame; xdot-frame; xdot-fluid
% d1=KVMB dashpot value, delta=equiv. KV damping ratio
m1=Mw;  %water mass
m2=Mf;  %frame mass
A=[0, 1, 0;
  -kf/m2,  -d1/m2,  d1/m2;
  0,  d1/m1,  -d1/m1];
%[V, lambda]=eig(A);
lambda=eig(A); 
%A=V*x*inv(V)
%eigenvalues from diagonal of x, lambda
%frame natural frequency
fnf=sqrt(kf/m2);  %rad/s
%select eigenvalues for frame component of motion
j=1;
jselect=zeros(2,1);
for jj=1:3
tst(jj)=abs(fnf-abs(imag(lambda(jj))));
end %next jj
target=min(tst);
for jj=1:3
if  tst(jj) == target  
jselect(j)=jj;
j=j+1;
end  %endif
end  %next jj
lambda1=lambda(jselect(1));
lambda2=lambda(jselect(2));
% compute KV equivalent damping ratio from 2 eigenvalues
delta=real(abs(lambda1+lambda2)/(2*sqrt(lambda1*lambda2)));
frequ=real(abs(imag(lambda2))/2/pi);
end  %endfunction


function [drv]=deriv(x)
%numerical derivative
ptrb=.1;
f1=KVMB(x-ptrb*x);
f2=KVMB(x+ptrb*x);
drv=(f2-f1)/(2*ptrb*x);
end %endfunction

function [x]=linesrch(dmin,dmax)
x1=dmin;
x2=dmax;
xm=(x2+x1)/2 ;  %midpoint of bracket
tol=1.E-19;
Fm=deriv(xm);
j=0;
%bisection search
while abs(Fm)>tol && j<100,
j=j+1;
F1=deriv(x1);
Fm=deriv(xm);
F2=deriv(x2);
if sign(F1)==sign(Fm) 
x1=xm;
F1=Fm;
xm=(x2+x1)/2;
else
x2=xm;
F2=Fm;
xm=(x2+x1)/2;
end  %endif
end %endwhile
if j== 100
msgbox('line search failed');
end
x=xm;
end %endfunction

%bracket determination function
function [da1,da2,db1,db2,dpeak]=bracket(dmin,dmax)
%find peak damping ratio
dpeak=linesrch(dmin,dmax);
da1=dmin;
da2=dpeak;
db1=dpeak;
db2=dmax;
end  %endfunction

%define inline objective functon for bisection search
function [F]=objf(d)
%d= KVMB dashpot value, drKV is observed KV damping ratio
global drKV
[delta,frequ]=KVMB(d);
F=delta-drKV;
end  %endfunction

function [x]=bisec(x1,x2)
xm=(x2+x1)/2 ;  %midpoint of bracket
tol=.0000001;
Fm=objf(xm);
j=0;
%bisection search
while abs(Fm)>tol && j<100 ,
j=j+1;
F1=objf(x1);
Fm=objf(xm);
F2=objf(x2);
if sign(F1)==sign(Fm) 
x1=xm;
F1=Fm;
xm=(x2+x1)/2;
else
x2=xm;
F2=Fm;
xm=(x2+x1)/2;
end  %endif
end %endwhile
if j~= 100
x=xm;
else
x=-1;
end
end %endfunction

%-------start of program-----------
%----------------------------------------------------------------------

%--------------------KVMB Analysis-----------------------
%set up bracket for search
global drKV=dratioKV;
KDmax=100; % max permeability m/s
KDmin=1.E-6;  % min permeability m/s
dmax=(A*L*rhow*9.81*umeanpor^2)/KDmin; %lower bound dashpot
dmin=(A*L*rhow*9.81*umeanpor^2)/KDmax; %upper bound dashpot
%find brackets, coupled and uncoupled search
 [da1,da2,db1,db2,dpeak]=bracket(dmin,dmax);
 tst=KVMB(dpeak);  %apex of curve =KVMB(da2)=KVMB(db1)
 drpksv=tst;
 if drKV>tst 
 KDa=0;
 KDb=0;
 drasv=0;
 drbsv=0;
 sigmaKDb=0;
 sigmaKDa=0;
 else
%do bisection search
 [xa]=bisec(da1,da2);
 dr=KVMB(xa);
 if xa==-1 
 dr=-1;
 end
 drasv=dr;
 tmp1=(A*L*rhow*9.81*umeanpor^2);
 KDa=tmp1/xa ;  %permeability m/s
 tmp2=-tmp1/(xa*xa);  %derivative of KDa wrt dashpot, xa
 tmp3=deriv(xa);   %derivative of damping ratio wrt dashpot
%stdev for permeability, KDa
 sigmaKDa=sqrt((tmp2/tmp3)^2*sigmadr^2); 
 [xb]=bisec(db1,db2);
 if xb==-1 
 dr=-1;
 end
 dr=KVMB(xb);
 drbsv=dr;
 tmp1=(A*L*rhow*9.81*umeanpor^2);
 KDb=tmp1/xb ;  %permeability m/s
 tmp2=-tmp1/(xb*xb);  %derivative of KDb wrt dashpot, xb
 tmp3=deriv(xb);      %derivative of damping ratio wrt dashpot
%stdev for permeability, KDb
 sigmaKDb=sqrt((tmp2/tmp3)^2*sigmadr^2); 
 end

%output
Tr=C2/C1*1000; % milliseconds relaxation time from wave equation coef.
msg0=sprintf('  SOLUTION (+/- 95 Percent Confidence)\n');
msg1=sprintf(...
'Freq=%.0f(Hz) Resonator_L=%.2f(m)\n'...
,freqdr,L);
msg2=sprintf(...
'Damping Ratios: Peak=%.6f Wave=%.6f (+/-%.5f)\n'...
,drpksv,dratioKV,sigmadr*1.96);
msg3=sprintf(...
'Coupled (b_case): DR=%.6f KD=%.5f(+/-%.4fm/s)\n'...
,drbsv,KDb,sigmaKDb*1.96);
msg4=sprintf(...
'UnCoupled (a_case): DR=%.6f KD=%.5f(+/-%.4fm/s)\n'...
,drasv,KDa,sigmaKDa*1.96);
msg5=sprintf(...
'Porosity: %.3f (+/-%.3f)\n',umeanpor,ustdpor*1.96);
msg6=sprintf(...
'Relaxation Time Tr=C2/C1=%.2f msec.',Tr);
MSG=[msg0,msg1,msg2,msg3,msg4,msg5,msg6]
msgbox([msg0,msg1,msg2,msg3,msg4,msg5,msg6]);

