%   |------------------------------------------------------------|
%   | $Id: SASW.m,v 1.2 2020/05/06 19:29:47 pm Exp $             |
%   | Basic Seismic Utilities (BSU)                              |
%   | LANGUAGE: octave                                           |
%   | AUTHOR: P. MICHAELS                                        |
%   | FUNCTION NAME: SASW.m                                      |
%   | DIRECTORY: BSUROOT/octave                                  |
%   |  revised for OCTAVE                                        |
%   |------------------------------------------------------------|
%   | purpose: Compute dispersion curve from cross correlation   |
%   |          of two geophone signals in same BSEGY file        |
%   |                                                            |
%   |------------------------------------------------------------|
%   | octave is an interpreter useful for scientifc calculations.|
%   | It is similar to MATLAB and may be obtained free from      |
%   | http:% www.gnu.org/software/octave                         |
%   |------------------------------------------------------------|
%   |  NOTICE:                                                   |
%   |  Copyright (c) 2017 Paul Michaels                          |
%   |  <pm@cgiss.boisestate.edu>                                 |
%   |  This program is free software; you can                    |
%   |  redistribute it and/or modify it under the terms          |
%   |  of the GNU General Public License as published            |
%   |  by the Free Software Foundation; either version           |
%   |  3 of the License, or (at your option) any later           |
%   |  version.  This program is distributed in the              |
%   |  hope that it will be useful, but                          |
%   |  WITHOUT ANY WARRANTY; without even the implied            |
%   |  warranty of MERCHANTABILITY or FITNESS FOR A              |
%   |  PARTICULAR PURPOSE.  See the GNU General Public           |
%   |  License for more details.                                 |
%   |  You should have received a copy of the GNU                |
%   |  General Public License along with this program;           |
%   |  if not, write to the Free Software Foundation,            |
%   |  Inc., 675 Mass Ave, Cambridge, MA 02139, USA.             |
%   |------------------------------------------------------------|
clear
pkg load signal
%fflush(stdout);
filen=input('enter file name ','s');
fp1=fopen(filen,'r');

%read *.seg file info:  ntraces=number of traces
%                       npts=number samples per trace
%                       fsamin=sample interval in seconds
[nbytes,ntraces,npts,fsamin]=segyinfo(fp1);

%read trace number jtrace by call to bsegin
%        data=vector of digitized samples
%        frange=source receiver separation
%        pic=first arrival time pick if any
%        rx,ry,rz=geophone coordinates
%        sx,sy,sz=source coordinates
%        sd=source depth, below sz


[data1,frange1]=bsegin(fp1,npts,1);
[data2,frange2]=bsegin(fp1,npts,2);
DX=abs(frange2-frange1);


%  control blow up plot parameters
fmin=5;       % minimum frequency to plot
fmax=50;      % maximum frequency to plot
cvelmin=0;    % minimum phase velocity to plot
cvelmax=500;  % maximum phase velocity to plot
prompt = {'Fmin (Hz)','Fmax (Hz)','Vmin (m/s)','Vmax (m/s)'};
dlg_title = 'Enter Parameter';
num_lines = 1;
defaultans = {num2str(fmin),num2str(fmax),num2str(cvelmin),num2str(cvelmax)};
cn = inputdlg(prompt,dlg_title,num_lines,defaultans);
frect = str2double(cn);
fmin=frect(1); fmax=frect(2); cvelmin=frect(3); cvelmax=frect(4);

%data set settings
xstr=sprintf('  %s \n %d traces \n tmax=%.3f sec.\n dt=%.5f (sec) dx=%.1f (m)',...
filen,ntraces,(npts-1)*fsamin,fsamin,DX);
%guidance on trace separation
% assume 100 m/s slow, abs(R2-R1) trace spacing should be < wavelength at fmax
vdesign=100; %about slow limit for soil velocities
fdesign=50;  %about limit for engineering seismic sources
dxdesign=vdesign/fdesign;
drdesign=dxdesign/DX; %recommended separation maximum between near and far traces
rstr=sprintf('\n Recommended Maximum Trace Separation in traces \n R2-R1=%d trace(s) ',...
int32(drdesign));
msgbox(strcat(xstr,rstr));
tmax=(npts-1)*fsamin;

prompt = {'Tmax','Trace Number R1','Trace Number R2'};
dlg_title = 'Enter Parameter';
num_lines = 1;
defaultans = {num2str(tmax),num2str(1),num2str(ntraces)};
cn = inputdlg(prompt,dlg_title,num_lines,defaultans);
frect = str2double(cn);
tmax=frect(1); jtrace=frect(2); ktrace=frect(3);

[datak,frangek,pick,rxk,ryk,rzk,sxk,syk,szk,sdk,spk,vpk,geoazik]=bsegin(fp1,npts,ktrace);
[dataj,frangej,picj,rxj,ryj,rzj,sxj,syj,szj,sdj,spj,vpj,geoazij]=bsegin(fp1,npts,jtrace);
dx=frangek-frangej;
fclose(fp1); 


%  maximum sample number to plot
jmax=tmax/fsamin+1;
if jmax > npts 
jmax=npts;
end;

%  setup time vector
tm=0:(npts-1);
tm=tm*fsamin;

%   compute cross spectrum
sj=fft(dataj);
sk=fft(datak);
if dx < 0
z=conj(sk).*sj; %  cross-spectral density
else
z=conj(sj).*sk; %  cross-spectral density
end
sec_leng=floor(npts/4);
sec_step=sec_leng/2;
overlap=.5;
psj=pwelch(dataj,sec_leng,overlap);
psk=pwelch(datak,sec_leng,overlap);
psjk=pwelch(dataj,datak,sec_leng,overlap,'cross');

%compute coherence estimate
cohrnc=(abs(psjk).^2)./(psj.*psk);
N=length(psj);
frq2=0:N-1;
frq2=frq2/(2*N*fsamin);

%set up frequency vector 
ndt=npts*fsamin;
frq=0:npts-1;
frq=frq/ndt;

% cross power spectrum, phase (Rad) and amplitude (dB)
GxyPhase=unwrap(angle(z));
dB=10*log10(abs(z)/max(z));

figure(1)
clf
subplot(311)
Nmxm=npts/2+1;
plot(frq(1:Nmxm),dB(1:Nmxm))
title('Cross Power Spectrum');
ylabel('Decibels');
%plot(frq,abs(z))
grid on
subplot(312)
plot(frq(1:Nmxm),GxyPhase(1:Nmxm))
ylabel('Radians');
grid on
subplot(313)
plot(frq2,cohrnc,'-r')
title('Coherence')
xlabel('Frequency Hz');
grid on
%=====================================
%  compute phase velocity blowup limits
npts=length(frq);
df=frq(2)-frq(1);
fshift=frq(1);
j1=floor((fmin-fshift)/df+1);
j2=floor((fmax-fshift)/df + 1);
if j1<1 
j1=1;
end
if j2>length(frq) 
j2=length(frq);
end

%  Compute Phase Velocity from unwrapped phase
Phz=GxyPhase*(180/pi);
for j=2:npts
T(j)=1/frq(j);
dT(j)=T(j)*(Phz(j)/360);
cvel(j)=abs(dx/dT(j)); %  meters/sec
end
figure(2)
subplot(2,1,1)
plot(frq(j1:j2),cvel(j1:j2))
axis([0,fmax,cvelmin,cvelmax]);
grid on
TL=sprintf('File: %s Trace: %d---%d  Phase Velocity ',filen,jtrace,ktrace);
title(TL)
xlabel('Frequency Hz');
ylabel('Velocity m/s');
subplot(2,1,2)
dfc=1/(sec_leng*fsamin);
j1c=floor(fmin/dfc+1);
j2c=floor(fmax/dfc+1);
plot(frq2(j1c:j2c),cohrnc(j1c:j2c),'-r')
axis([0,fmax,0,1.0]);
grid on
title('Coherence')
xlabel('Frequency Hz')

