%   |------------------------------------------------------------|
%   | $Id: invR1.m,v 1.1.1.1 2019/04/29 22:51:20 pm Exp $        |
%   | Basic Seismic Utilities (BSU)                              |
%   | LANGUAGE: octave                                           |
%   | AUTHOR: P. MICHAELS                                        |
%   | FUNCTION NAME: invR1.m                                     |
%   | DIRECTORY: BSUROOT/octave                                  |
%   |  revised for OCTAVE                                        |
%   |------------------------------------------------------------|
%   | Invert rayleigh wave given bvax results                    |
%   | uses SVD inverse, set p=number of singular values          |
%   |                                                            |
%   |Starting model: model.txt is a file with 3 rows             |
%   |Example: for nlay=3 vi=shear velocity, zi=depth layer top   |
%   | nlay                                                       |
%   | v1 v2 v3                                                   |
%   | z1 z2 z3                                                   |
%   |------------------------------------------------------------|
%   | octave is an interpreter useful for scientifc calculations.|
%   | It is similar to MATLAB and may be obtained free from      |
%   | http://www.gnu.org/software/octave                         |
%   |------------------------------------------------------------|
%   |  Note: requires disper.oct (run build_disper_oct to build) |
%   |------------------------------------------------------------|
%   |  NOTICE:                                                   |
%   |  Copyright (c) 2017 Paul Michaels                          |
%   |  <pm@cgiss.boisestate.edu>                                 |
%   |  This program is free software; you can                    |
%   |  redistribute it and/or modify it under the terms          |
%   |  of the GNU General Public License as published            |
%   |  by the Free Software Foundation; either version           |
%   |  3 of the License, or (at your option) any later           |
%   |  version.  This program is distributed in the              |
%   |  hope that it will be useful, but                          |
%   |  WITHOUT ANY WARRANTY; without even the implied            |
%   |  warranty of MERCHANTABILITY or FITNESS FOR A              |
%   |  PARTICULAR PURPOSE.  See the GNU General Public           |
%   |  License for more details.                                 |
%   |  You should have received a copy of the GNU                |
%   |  General Public License along with this program;           |
%   |  if not, write to the Free Software Foundation,            |
%   |  Inc., 675 Mass Ave, Cambridge, MA 02139, USA.             |
%   |------------------------------------------------------------|
clear
function [grho,gmu,glame]=gparms(rho,mu,lame,zi,nlay,depth)
      l1=1;
      for j=1:nlay
      	if(depth>=zi(j)); 
      	l1=j ;
	end
      end
      if(l1<nlay) 
      l2=l1+1;
      dz=zi(l2) - zi(l1);
      sloper=(rho(l2)-rho(l1))/dz;
      slopel=(lame(l2)-lame(l1))/dz;
      slopem=(mu(l2)-mu(l1))/dz;
      z=depth-zi(l1);
      grho=sloper*z + rho(l1);
      gmu=slopem*z + mu(l1);
      glame=slopel*z + lame(l1);
      else
      grho=rho(nlay);
      gmu=mu(nlay);
      glame=lame(nlay);
      end
endfunction

function [alpha,beta1]=e2v(Rho,Mu,Lame)
beta1=sqrt(Mu./Rho);
alpha=sqrt(Lame./Rho+2*beta1^2);
endfunction

function [c]=dcal(Nfreq,nlay,zi,Alpha,Beta,rho,deltz,Freq)
%compute calculated data
for j=1:length(zi)
mu(j,1)=rho(j)*Beta(j)^2; % shear modulus
lame(j,1)=rho(j)*(Alpha(j)^2-2.00*Beta(j)^2); % Lame's constant
end
for j=1:Nfreq
frq1=Freq(j);
maxmod=1;
cvel=zeros(maxmod,1);
cvel=disper(nlay,rho,mu,lame,zi,deltz,frq1,maxmod,cvel);
c(j)=cvel(1);
end
endfunction

%compute derivatives wrt Beta
function [dcdb]=DCDB(Nfreq,nlay,zi,Alpha,Beta,rho,pertb,deltz,Freq)
NB=length(Beta);
for j=1:NB
pb=pertb*Beta(j);
svbeta=Beta(j);
Beta(j)=svbeta+pb;
cp=dcal(Nfreq,nlay,zi,Alpha,Beta,rho,deltz,Freq);
Beta(j)=svbeta-pb;
cm=dcal(Nfreq,nlay,zi,Alpha,Beta,rho,deltz,Freq);
derv=(cp-cm)/(2*pb);
dcdb(:,j)=derv;
Beta(j)=svbeta;
end %next j
endfunction

%compute derivatives wrt zi
function [dcdz]=DCDZ(Nfreq,nlay,zi,Alpha,Beta,rho,pertb,deltz,Freq)
NZ=length(zi);
for j=2:NZ
pb=pertb*zi(j);
svz=zi(j);
zi(j)=svz+pb;
cp=dcal(Nfreq,nlay,zi,Alpha,Beta,rho,deltz,Freq);
zi(j)=svz-pb;
cm=dcal(Nfreq,nlay,zi,Alpha,Beta,rho,deltz,Freq);
derv=(cp-cm)/(2*pb);
dcdz(:,j-1)=derv;
zi(j)=svz;
end %next j
endfunction

function [A,B,Z,R]=plotvel(iwin,clrsw,zi,deltz,rho,mu,lame,nlay)
%plot velocity model
nz=floor(zi(nlay)/deltz+.5);
zo=zi(1);
for j=1:2:nz*2
z1=zo+deltz;
depth=(z1+zo)/2;
[grho,gmu,glame]=gparms(rho,mu,lame,zi,nlay,depth);
[alpha,beta1]=e2v(grho,gmu,glame);
A(j,1)=alpha;
B(j,1)=beta1;
Z(j,1)=zo;
R(j,1)=grho;
A(j+1,1)=alpha;
B(j+1,1)=beta1;
Z(j+1,1)=z1;
R(j+1,1)=grho;
zo=z1;
end %next j
figure(iwin)
hold on
if clrsw==0 
plot(A,-Z,'r') %p-wave velocity
plot(B,-Z,'b') %s-wave velocity
plot(R,-Z) %density
else
plot(A,-Z) %p-wave velocity
plot(B,-Z) %s-wave velocity
end
%xtitle('Model','Velocity','Depth');
grid;
endfunction

function [VSave]=vsave(zi,deltz,rho,mu,lame,nlay)
nz=floor(zi(nlay)/deltz+.5);
zo=zi(1);
tsum=0;
zmax=zi(nlay);
for j=1:2:nz*2
z1=zo+deltz;
depth=(z1+zo)/2;
[grho,gmu,glame]=gparms(rho,mu,lame,zi,nlay,depth);
[alpha,beta1]=e2v(grho,gmu,glame);
tsum=tsum+deltz/beta1;
zo=z1;
end %next j
VSave=zmax/tsum;
endfunction


%====================begin program====================
igo=1;
while (igo)
c=exist('disper.oct');
 if c == 0
 c2=exist('build_disper_oct');
 c3=exist('wrapper.cpp');
 c4=exist('rwv.f');
   if c2 != 0 && c3 != 0 && c4 !=0
   T=sprintf('Requires disper.oct \nTo run build_disper_oct, CLICK OK');
   msgbox(T);
   system('build_disper_oct');
 else
   T1=sprintf('ABORT\n');
   if c2 ==0 T2=sprintf('build_disper_oct missing\n'); else T2=' '; end
   if c3 ==0 T3=sprintf('wrapper.cpp missing\n'); else T3=' '; end
   if c4 ==0 T4=sprintf('rwv.f missing'); else T4=' '; end 
   T=strcat(T1,T2,T3,T4);
   msgbox(T);
break
end
end

%Initial Model Parameters

%nlay=5;
%Beta=[ 194;104.35; 91.457; 214.78; 350.];  % shear velocities
%zi=[    0; 1.14; 2.320; 12.79; 30.0;]; % depth control points
Query=sprintf('enter file with initial model \n example: model.txt ');
ifile=input(Query,'s');
invar=ifile(1: findstr(ifile,'.')-1);
fp1=fopen(ifile);
[invar]=fscanf(fp1,'%f');
nlay=invar(1)
Beta(1:nlay)=invar(2:1+nlay)
zi(1:nlay)=invar(2+nlay:1+nlay+nlay)
fclose(fp1)


msg0='Fixed Vp/Vs or Fixed Vp';
BTN1='Vp/Vs ';
BTN2='Vp=fix';
TITLE='Choose Vp';
BTN=questdlg(msg0,TITLE,BTN1,BTN2,BTN1);

if BTN == 'Vp=fix'
  alphafixed=1700;
  rhomoist=2100;

prompt = {'Fixed Vp','Density kg/m3'};
dlg_title = 'Enter Fixed Properties';
num_lines = 1;
defaultans = {'1700','2100'};
cn = inputdlg(prompt,dlg_title,num_lines,defaultans);
frect = str2double(cn);
alphafixed=frect(1); rhomoist=frect(2);
  for j=1:nlay
  Alpha(j,1)=alphafixed;  %alternative to Poisson's Ratio
  rho(j)=rhomoist; %Densities
  end
else
%alpha and rho settings
sigma=1/3;  %Poisson's Ratio to set Alpha
alphafixed=1700; %alternative to sigma
Gs=2.67;    %grain density
npor=.3;    %porosity
evr=npor/(1-npor); %void ratio
Sat=1.0;  %degree water saturation

%get arguments
sigmatxt=sprintf('%.2f',sigma);
Gstxt=sprintf('%.2f',Gs);
nportxt=sprintf('%.2f',npor);
Sattxt=sprintf('%.2f',Sat);
prompt = {'Poisson Ratio','Solid g/cc','porosity','Degree Water Sat.'};
dlg_title = 'Enter Soil Properties for Vp/Vs';
num_lines = 1;
defaultans = {sigmatxt,Gstxt,nportxt,Sattxt};
cn = inputdlg(prompt,dlg_title,num_lines,defaultans);
frect = str2double(cn);
sigma=frect(1); Gs=frect(2); npor=frect(3); Sat=frect(4);

evr=npor/(1-npor); %void ratio
rhomoist=((Gs+Sat*evr)*1000)/(1+evr);  %mass density soil
alphaObeta=sqrt( 2*(1-sigma) / (1-2*sigma));
for j=1:nlay
Alpha(j,1)=Beta(j)*alphaObeta; %P-wave velocities
rho(j)=rhomoist; %Densities
end
end

%=============Vp choice=============================
if BTN == 'Vp=fix'
msg1=sprintf('Vp=%.0f m/s \nDen=%.0f kg/m3',alphafixed,rhomoist);
msgbox(msg1);
else
msg1=sprintf('Vp/Vs = %.2f \nDen=%.0f kg/m3',alphaObeta,rhomoist);
msgbox(msg1);
end
 

% interval layer thickness and other parameters
%get arguments
prompt = {'p (singular values)','deltz','locksw (see code)','iterations'};
dlg_title = 'Enter Parameters';
num_lines = 1;
defaultans = {num2str(3),num2str(.1),num2str(3),num2str(2)};
cn = inputdlg(prompt,dlg_title,num_lines,defaultans);
frect = str2double(cn);
p=frect(1); deltz=frect(2); locksw=frect(3); itmax=frect(4)

%deltz=.2; 
maxmod=1;  %fundamental mode only
%p=3;  %number of singular values to include
rowyes=1;  %include row weighting? 1=yes
%locksw=0;  % free bottom control, velocity and depth
%locksw=1;  % lock bottom depth, free bottom velocity 
%locksw=2;  % lock bottom velocity, free bottom depth 
%locksw=3;  % lock both bottom velocity and depth 

%observed data
FILE='bvax.his';
fp1=fopen(FILE,'rt');
data=fscanf(fp1,'%f %f %f %f %f %f \n',[6,Inf]);
fclose(fp1);
Freq=data(1,:)';
dobs=data(2,:)';
em=data(3,:)*1.96'; %95 percent conf
Nfreq=length(Freq);
ep=em;

pertb=.0001;
%pertz=1;

%row weighting
% set row weighting?
n=Nfreq;
wr=eye(n,n);
%rowyes=1;
if (rowyes==1) 
  for j=1:n
   wr(j,j)=1/(em(j)/1.96);  %weight by reciprocal stdv 
  end
end
% plot observed data
figure(1)
clf
hold on 
errorbar(Freq,dobs,em)
grid
title('Rayleigh Wave');
xlabel('Frequency Hz');
ylabel('Phase Velocity (m/s)');
pause(1);
%start loop
%itmax=2;
for iter=1:itmax

%compute calculated data
c=dcal(Nfreq,nlay,zi,Alpha,Beta,rho,deltz,Freq);
e=dobs-c';
mk=[Beta';zi(2:length(zi))'];
if (rowyes==1) 
 wre=wr*e;
 esv(iter)=wre'*wre;
 else
 esv(iter)=e'*e;
end
LsQe=sqrt(esv(iter)/Nfreq)  %list LSQE
hold on
plot(Freq,c,'g');
pause(1);
fflush(stdout);
[dcdb]=DCDB(Nfreq,nlay,zi,Alpha,Beta,rho,pertb,deltz,Freq);
[dcdz]=DCDZ(Nfreq,nlay,zi,Alpha,Beta,rho,pertb,deltz,Freq);
G=[dcdb dcdz];
[n,m]=size(G);

%column weighting
w=eye(m,m);
for j=1:m
 XXX=mean(abs(G(:,j)));
  if (XXX ~= 0.) 
  w(j,j)=1/XXX;
  end
end
wrG=wr*G;
wre=wr*e;

Gw=wrG*w;

[U,S,V]=svd(Gw);
for i=1:n
for j=1:p
Up(i,j)=U(i,j);
end
end
for i=1:m
for j=1:p
Vp(i,j)=V(i,j);
end
end
for i=1:p
for j=1:p
Sp(i,j)=S(i,j);
end
end
eigv=svd(Gw);

%solve for parameters
H=w*Vp*inv(Sp)*Up'*wr;
dm=H*e;

%damping changes
dampchg=1.;
mk=mk+dm*dampchg;

%update parameters
 if locksw==0 || locksw==1 
  jendb=length(Beta);    % length(Beta)   frees bottom velocity
 else
  jendb=length(Beta)-1; % length(Beta)-1 locks bottom velocity
 end

%round off depth control points to nearest 1/100
for j=jendb+1:length(mk)
  MK=floor(mk(j)*100.);
  mk(j)=MK/100;
end

minvelset=75;
for j=1:jendb
  if mk(j) >minvelset 
  Beta(j)=mk(j);
  else
  Beta(j)=minvelset;  %set minimum S-velocity
  end
  if Beta(j)>Beta(nlay) 
  Beta(j)=Beta(nlay);  %set maximum S-velocity as last control
  end
end
jstart=length(Beta)+1;
if locksw==0 || locksw==2 
jend=length(mk);    %jend=length(mk)   frees bottom depth
else
jend=length(mk)-1;  %jend=length(mk)-1 locks bottom depth
end
k=2;
if locksw==2 || locksw==0  
for j=jstart:jend
  if mk(j)>zi(k-1)  
  zi(k)=mk(j);  %only update depth control if increasing monotonically
  end
k=k+1;
end
else
for j=jstart:jend
  if mk(j)>zi(k-1) && mk(j)<zi(k+1) 
  zi(k)=mk(j);  %only update depth control if increasing monotonically
                %and bound below bottom value
  end
k=k+1;
end

end

if BTN == 'Vp/Vs '
 for j=1:nlay
 Alpha(j)=Beta(j)*alphaObeta;
 end
end


end %next iter
%plot(Freq,c,'r');
%====================================================================
%save singular values
[nrow,ncol]=size(S);
for j=1:ncol
L(j)=S(j,j);
end

%error analysis
em1stdv=em'./1.96;  %68% confidence 1 stdev
em1var=em1stdv.*em1stdv;
Cd=eye(n,n); %data covariance matrix
for j=1:n
Cd(j,j)=Cd(j,j)*em1var(j);
end
Cm=H*Cd*H'; %using whatever em %conf is.
for k=1:m
mkeb(k,1)=(sqrt(Cm(k,k)))*1.96; %95% confidence
end

%compute calculated data at solution
c=dcal(Nfreq,nlay,zi,Alpha,Beta,rho,deltz,Freq);
e=dobs-c';
mk=[Beta';zi(2:length(zi))'];
if (rowyes==1) 
 wre=wr*e;
 esv(itmax+1)=wre'*wre;
 else
 esv(itmax+1)=e'*e;
end
LsQe=sqrt(esv(itmax+1)/Nfreq)  %list LSQE
hold on
plot(Freq,c,'r');
% print('dispcrv.fig','-dfig')
pause(1);

% Error bars given constraints assignment
% constraints on bottom depth, else none
if (locksw==3 || locksw==1)
   deptheb=[0.0;mkeb((nlay+1):(m-1));0.0];
  else
   deptheb=[0.0;mkeb((nlay+1):m)];
end
% constraints on bottom velocity, else none
if (locksw==2 || locksw==3)
   betaeb=[mkeb(1:(nlay-1));0];
  else
   betaeb=[mkeb(1:nlay)];
end

% print results character strings
T1a=sprintf('Vs=%.1f ',Beta);
T1b=sprintf('(+/-%.2f) ',mkeb(1:nlay));
T2a=sprintf('Zi=%.3f ',zi);
T2b=sprintf('(+/-%.1f) ',deptheb);
T3=sprintf('deltz=%.3f Lsqe=%.3e',deltz,sqrt(esv(itmax+1)/Nfreq));
SUBTITLE=sprintf('%s \n %s \n %s',T1a,T2a,T3);

Betasave=Beta;
zisave=zi;
% save fast limit model to a file
Beta=Betasave+betaeb';
zi=zisave-deptheb';
save('-ascii','fast.txt','nlay','Beta','zi');
% save slow limit model to a file
Beta=Betasave-betaeb';
zi=zisave+deptheb';
save('-ascii','slow.txt','nlay','Beta','zi');
%restore and save solution to a file
Beta=Betasave;
zi=zisave;
save('-ascii','solution.txt','nlay','Beta','zi');

% save solution for plotting in gnuplot
tmp=[Freq,c'];
save dispcrv.dat -ascii tmp
tstamp=strftime('%Y.%j %H:%M:%S',localtime (time));
fp1=fopen('dispcrv.gp','wt');
fprintf(fp1,'# Generated by invR1.m  \n')
fprintf(fp1,'# Date: %s \n',tstamp)
fprintf(fp1,'# %s \n',T1a)
fprintf(fp1,'# %s \n',T1b)
fprintf(fp1,'# %s \n',T2a)
fprintf(fp1,'# %s \n',T2b)
fprintf(fp1,'# %s \n',T3)
fprintf(fp1,'                        \n')
fprintf(fp1,'set mxtics \n')
fprintf(fp1,'set mytics \n')
fprintf(fp1,'set grid \n')
fprintf(fp1,'set title "Solution" \n')
fprintf(fp1,'set xlabel "Frequency Hz" \n')
fprintf(fp1,'set ylabel "Velocity m/s" \n')
fprintf(fp1,'plot "dispcrv.dat" u 1:2 w lines lw 2 t "Calculated", "bvax.his" using 1:2:3 with errorbars pt 6 lc 0 lt 1 t "Observed" \n');
fprintf(fp1,'set terminal png size 1024,768 \n')
fprintf(fp1,'set output "dispcrv.png"\n')
fprintf(fp1,' set pointsize 1.5       \n')
fprintf(fp1,'replot  \n')
fprintf(fp1,'set terminal pdf color enhanced size 6,6\n')
fprintf(fp1,'set output "dispcrv.pdf"                        \n')
fprintf(fp1,' set pointsize .9       \n')
fprintf(fp1,'replot                  \n')
fclose(fp1);


%errbar(Freq,dobs,em,ep)
%xtitle(['Rayleigh Wave';T1;T2;T3],'Frequency (Hz)','Phase Velocity (m/s)');

%compute elastic properties
for j=1:length(zi)
mu(j,1)=rho(j)*Beta(j)^2; %shear modulus
lame(j,1)=rho(j)*(Alpha(j)^2-2.00*Beta(j)^2); %Lame's constant
end

[VSave]=vsave(zi,deltz,rho,mu,lame,nlay)


%plot velocity and density model
iwin=2;
figure(iwin);
clf;
clrsw=0; %color switch
[A,B,Z,R]=plotvel(iwin,clrsw,zi,deltz,rho,mu,lame,nlay);
hold on
plot(Beta,-zi,'markersize',15,'o');  %plot control points on S-velocity
%display error analysis bounds
xlabel(SUBTITLE);
title('Soil Profile');
ylabel('Depth m');

clear tmp
tmp=[A,B,Z,R];
save "model.dat" -ascii tmp
clear tmp
tmp=[Beta',zi'];
save "control.dat" -ascii tmp
Zmax=(floor(max(zi)/10.0))*10;
fp1=fopen('model.gp','wt');
fprintf(fp1,'# Generated by invR1.m  \n')
fprintf(fp1,'# Date: %s \n',tstamp)
fprintf(fp1,'# %s \n',T1a)
fprintf(fp1,'# %s \n',T1b)
fprintf(fp1,'# %s \n',T2a)
fprintf(fp1,'# %s \n',T2b)
fprintf(fp1,'# %s \n',T3)
fprintf(fp1,'                        \n')
fprintf(fp1,'set mxtics 5 \n')
fprintf(fp1,'set mytics 5 \n')
fprintf(fp1,'set grid \n')
fprintf(fp1,'set title "Model VSave=%.1f m/s"  \n',VSave)
fprintf(fp1,'set xlabel "S-Wave Velocity m/s" \n')
fprintf(fp1,'set ylabel "Depth m" \n')
fprintf(fp1,'set pointsize 2.0       \n')
fprintf(fp1,'plot [0:500]  [%.0f:0]  "model.dat" using 2:3 w lines t "S-velocity", "control.dat" using 1:2 w points ls 6 lc 0 t "Control" \n ',Zmax);
fprintf(fp1,'set terminal png size 1024,768 \n')
fprintf(fp1,'set output "model.png"\n')
fprintf(fp1,' set pointsize 1.5       \n')
fprintf(fp1,'replot  \n')
fprintf(fp1,'set terminal pdf color enhanced size 6,6\n')
fprintf(fp1,'set output "model.pdf"                        \n')
fprintf(fp1,' set pointsize .9       \n')
fprintf(fp1,'replot                  \n')

fclose(fp1);

% Plus and Minus Limits
mkP=mk+mkeb;
BetaP=mkP(1:length(Beta));
ziP(2:length(zi))=mkP(length(Beta)+1:length(mk));
%compute elastic properties
for j=1:length(zi)
muP(j,1)=rho(j)*BetaP(j)^2; %shear modulus
lameP(j,1)=rho(j)*(Alpha(j)^2-2.00*BetaP(j)^2); %Lame's constant
end
iwin=2;
clrsw=1;
%[A,B,Z,R]=plotvel(iwin,clrsw,ziP,deltz,rho,muP,lameP,nlay);
mkM=mk-mkeb;
BetaM=mkM(1:length(Beta));
ziM(2:length(zi))=mkM(length(Beta)+1:length(mk));
%compute elastic properties
for j=1:length(zi)
muM(j,1)=rho(j)*BetaM(j)^2; %shear modulus
lameM(j,1)=rho(j)*(Alpha(j)^2-2.00*BetaM(j)^2); %Lame's constant
end
iwin=2;
%[A,B,Z,R]=plotvel(iwin,clrsw,ziM,deltz,rho,muM,lameM,nlay);
hold off


%wavelength plot
wl=dobs./Freq;
wlc=c'./Freq;
figure(3)
clf
hold on
plot(dobs,-wl,'o','markersize',5)
errorbar(dobs,-wl,em,em,0,0,'~>')
plot(c,-wlc,'r')
grid
xlabel(SUBTITLE);
ylabel('Wave Length m');
title('Velocity m/s');
hold off
clear tmp
tmp=[dobs,wl,c',wlc,em'];
save "wavelen.dat" -ascii tmp
Wmax=(floor(max(wl)/10.0))*10;


fp1=fopen('wavelen.gp','wt');
fprintf(fp1,'# Generated by invR1.m  \n')
fprintf(fp1,'# Date: %s \n',tstamp)
fprintf(fp1,'# %s \n',T1a)
fprintf(fp1,'# %s \n',T1b)
fprintf(fp1,'# %s \n',T2a)
fprintf(fp1,'# %s \n',T2b)
fprintf(fp1,'# %s \n',T3)
fprintf(fp1,'                        \n')
fprintf(fp1,'set mxtics 5 \n')
fprintf(fp1,'set mytics 5 \n')
fprintf(fp1,'set grid \n')
fprintf(fp1,'set title "Wavelength" \n')
fprintf(fp1,'set xlabel "S-Wave Velocity m/s" \n')
fprintf(fp1,'set ylabel "Wavelength m" \n')
fprintf(fp1,'plot [0:500] [%.0f:0] "wavelen.dat" using 1:2:5 w xerrorbars ls 6 lc 0 t "observed", "" using 3:4 w lines lw 2 lc 1 lt 1 t "calculated" \n ',Wmax);
fprintf(fp1,'set terminal png size 1024,768 \n')
fprintf(fp1,'set output "wavelen.png"\n')
fprintf(fp1,' set pointsize 1.5       \n')
fprintf(fp1,'replot  \n')
fprintf(fp1,'set terminal pdf color enhanced size 6,6\n')
fprintf(fp1,'set output "wavelen.pdf"                        \n')
fprintf(fp1,' set pointsize .9       \n')
fprintf(fp1,'replot                  \n')

fclose(fp1);


figure(4)
clf
plot(L,'-r')
hold on
plot(L,'o')
sval=sprintf('%.1E  ',L);
pval=sprintf(' p= %d ',p);
TL=strcat('Singular Values= ',sval,pval)
title(TL);
ylabel('Value')
plot([p,p],[0,max(L)],'-b')
jsv=1:length(L);
clear tmp;
tmp=[jsv',L'];
save "sval.dat" -ascii tmp

fp1=fopen('singlval.gp','wt');
fprintf(fp1,'# Generated by invR1.m  \n')
fprintf(fp1,'# Date: %s \n',tstamp)
fprintf(fp1,'# %s \n',T1a)
fprintf(fp1,'# %s \n',T1b)
fprintf(fp1,'# %s \n',T2a)
fprintf(fp1,'# %s \n',T2b)
fprintf(fp1,'# %s \n',T3)
fprintf(fp1,'# P=%d\n',p)
fprintf(fp1,'                        \n')
fprintf(fp1,'set mxtics 5 \n')
fprintf(fp1,'set mytics 5 \n')
fprintf(fp1,'set grid \n')
fprintf(fp1,'set title "%s" \n',TL)
fprintf(fp1,'set xlabel "Number" \n')
fprintf(fp1,'set ylabel "Singular Value" \n')
fprintf(fp1,'plot "sval.dat" using 1:2 w linespoints ls 2 lc 1 lt 6 t "singular values"\n ')
fprintf(fp1,'set terminal png size 1024,768 \n')
fprintf(fp1,'set output "sval.png"\n')
fprintf(fp1,' set pointsize 1.5       \n')
fprintf(fp1,'replot  \n')
fprintf(fp1,'set terminal pdf color enhanced size 6,6\n')
fprintf(fp1,'set output "sval.pdf"                        \n')
fprintf(fp1,' set pointsize .9       \n')
fprintf(fp1,'replot                  \n')

fclose(fp1);












igo=0
end %endwhile
