%   |------------------------------------------------------------|
%   | $Id: kdKVMBscan.m,v 1.1.1.1 2019/04/29 22:51:20 pm Exp $   |
%   | Basic Seismic Utilities (BSU)                              |
%   | LANGUAGE: octave                                           |
%   | AUTHOR: P. MICHAELS                                        |
%   | FUNCTION NAME: kdKVMBscan.m                                |
%   | DIRECTORY: BSUROOT/octave                                  |
%   |  revised for OCTAVE                                        |
%   |------------------------------------------------------------|
%   |2DOF System (water and frame) simulated by                  |
%   | Scan for different damping values                          |
%   |18 August 2005 Made Kd units m/s throughout                 |
%   |  Plots:  Kd (m/s) vs. KV damping ratio                     |
%   |          Pore Dia (mm) vs. KV damping ratio                |
%   | damping ratio=c/(2*sqrt(k*m))                              |
%   |   Based on original Scilab Code                            |
%   |   soil-norm-scan-eig.sci         2DOF->1DOF mapping        |
%   |                                                            |
%   |------------------------------------------------------------|
%   | octave is an interpreter useful for scientifc calculations.|
%   | It is similar to MATLAB and may be obtained free from      |
%   | http://www.gnu.org/software/octave                         |
%   |------------------------------------------------------------|
%   |  NOTICE:                                                   |
%   |  Copyright (c) 2017 Paul Michaels                          |
%   |  <pm@cgiss.boisestate.edu>                                 |
%   |  This program is free software; you can                    |
%   |  redistribute it and/or modify it under the terms          |
%   |  of the GNU General Public License as published            |
%   |  by the Free Software Foundation; either version           |
%   |  3 of the License, or (at your option) any later           |
%   |  version.  This program is distributed in the              |
%   |  hope that it will be useful, but                          |
%   |  WITHOUT ANY WARRANTY; without even the implied            |
%   |  warranty of MERCHANTABILITY or FITNESS FOR A              |
%   |  PARTICULAR PURPOSE.  See the GNU General Public           |
%   |  License for more details.                                 |
%   |  You should have received a copy of the GNU                |
%   |  General Public License along with this program;           |
%   |  if not, write to the Free Software Foundation,            |
%   |  Inc., 675 Mass Ave, Cambridge, MA 02139, USA.             |
%   |------------------------------------------------------------|

clear

%damping values as fraction of spring
Rk(1)=.0000001;
for j=2:105
Rk(j)=Rk(j-1)*1.2;
end %next j
npts=length(Rk);

isw1=1;
icase=0;
Gs=2.67;     %specific gravity of solid
rhow=1000;  %water mass density
rhof=rhow*Gs; %frame mass density
Ah=1;  %cross-section area 
vs=200; %shear wave velocity (doesn't matter, L adjusted)
no=0.3;  %reference porosity for Gf, L computation
muwater=1.12E-3;  %mks units viscosity Pa*s


% set type of plot Hydraulic Conductivity or Pore Diameter
MSG='Choose Horizontal Axis';
TITLE='Select';
BTN1='Hyd.Cond'; BTN2='Pore Dia'; DEFAULT=BTN1;
BTN=questdlg(MSG,TITLE,BTN1,BTN2,DEFAULT);
if BTN==BTN1
biotkd=0;
else
biotkd=1;
end

while isw1==1 ,
icase=icase+1;

%get arguments
prompt = {'Frequency','n (porosity)'};
dlg_title = 'Enter Parameters';
num_lines = 1;
defaultans = {'50','.25'};
cn = inputdlg(prompt,dlg_title,num_lines,defaultans);
frect = str2double(cn);
fn=frect(1); n=frect(2);
Kdmax=0.1;
omegan=pi*2*fn;
Gf=vs*vs*(rhof*(1-no));  %shear modulus frame
L=sqrt(Gf/((omegan^2)*(1-no)*Gs*rhow));  %length of sample
kf=(Gf*Ah)/L; %spring stiffness


%note: does not matter what Vs is since resonant frequency
%is specified, and L is adjusted to give the desired
% resonant frequency

e=n/(1-n);  %void ratio
mw=n*Ah*L*rhow;  %mass of water
mf=Ah*L*(1-n)*Gs*rhow; %mass of frame

m1f=mf+mw;  %total mass


for kk=1:length(Rk)
               %solve for Kd, darcy hydraulic conductivity
               %that corresponds to this df=dashpot kg/s
df=Rk(kk)*kf; %dashpot value for this case of scan
dfsv(kk)=df; %save dashpot kg/s
Kd(kk)=(Ah*L*n^2*rhow*9.81)/df;  % units m/s hydraulic conductivity
%Kd(kk)=Kd(kk)*100; %units cm/s
%pore diameter depends on viscosity
pordia(kk)=sqrt((Kd(kk)*32*muwater)/(9.81*n*rhow));
biot(kk)=df/(Ah*L);

%eigenvalues 2DOF System
A=[ 0, 1, 0;
  -kf/mf, -df/mf, +df/mf;
  0,  +df/mw , -df/mw];

%[V, delta]=eig(A);
delta=eig(A); 

%find real eigenvalue, discard
%imaginary parts
ip(1)=imag(delta(1));
ip(2)=imag(delta(2));
ip(3)=imag(delta(3));
%real parts
rp(1)=real(delta(1));
rp(2)=real(delta(2));
rp(3)=real(delta(3));
j1=1;
for j=1:3
if (abs(ip(j))>1.E-3) 
ev(j1)=rp(j)+i*ip(j);
j1=j1+1;
end  %endif
end % next j

%equivalent dashpot and spring
sk(kk)=(real((ev(1)*ev(2))))*mf;  %equivalent spring
sc(kk)=(-real((ev(1)+ev(2))))*mf; %equivalent dashpot
svfo(kk)=sqrt(sk(kk)/mf)/(pi*2);  %equivalent natural frequency hz
end  %next kk

%normalize to damping ratio
ndfsv=dfsv./(2*sqrt(kf*m1f));
nsc=sc./(2*sqrt(sk*mf));
%fixed 16Aug05 damping ratio definitions, 2 in denom

%determine subscript for plot
for kk=1:npts      
if Kd(kk)<Kdmax 
jstart=kk;
break;
end %endif
end %next kk

if icase==1 
%xset('window',1);
%mratio=mf/mw; %mass ratio, frame to water
%mtxt=sprintf("Mf/Mw=%.0f",mratio);
mtxt=sprintf('n=%.2f %.0f Hz',n,fn);
 if biotkd==0 
  loglog(Kd(jstart:npts),nsc(jstart:npts))
  axis([1e-6,.1,1e-5,1.0]);
  title('Equivalent KV Damping Ratio');
  xlabel('Hydraulic Conductivity m/s');
  ylabel('Damping Ratio KV');
  hold on
  grid on
  grid minor off
 else
%  loglog(biot(jstart:npts),nsc(jstart:npts))
%  title('Equivalent KV Damping Ratio');
%  xlabel('Biot b (kg/s per m^3)');
%  ylabel('Damping Ratio KV')
  loglog(pordia(jstart:npts)*1000,nsc(jstart:npts))
  axis([1e-3,1.,1e-6,1.0]);
  title('Equivalent KV Damping Ratio');
  xlabel('Pore Dia (mm)');
  ylabel('Damping Ratio KV')
  hold on
  grid on
  grid minor off
 end
msgbox('click on label location');
[xp,yp]=ginput(1);
text(xp,yp,mtxt);
else
mtxt=sprintf('n=%.2f %.0f Hz',n,fn);
 if biotkd==0 
  loglog(Kd(jstart:npts),nsc(jstart:npts))
  drawnow
  hold on
  grid on
  grid minor off
 else
%  loglog(biot(jstart:npts),nsc(jstart:npts))
  loglog(pordia(jstart:npts)*1000,nsc(jstart:npts))
  drawnow
  hold on
  grid on
  grid minor off
 end
msgbox('click on label location');
[xp,yp]=ginput(1);
text(xp,yp,mtxt);
end

%AAA=[nsc(jstart:npts)',pordia(jstart:npts)*1000',Kd(jstart:npts)'];
%fp1=fopen('water.dat','w');
%lbl=sprintf('Freq=%d Porosity=%f \n',fn,n);
%fprintf(fp1,'%s\n',lbl);
%for jj=1:npts
%fprintf(fp1,'14.7%E 14.7%E 14.7%E',AAA(jj,1:3))
%end
%fclose(fp1);

%Do an another ?
nc=questdlg('Continue?','Next Case','Yes','No','Yes');
switch nc
   case{'Yes'}
    isw1=1;
   case{'No'}
    isw1=0;
end

end %endwhile isw1

