c    $Id: rwv.f,v 1.1.1.1 2019/04/29 22:51:20 pm Exp $
c  |------------------------------------------------------------|
c  | rwv.f                                   date:  18 Nov 08   |
c  | author: p. michaels                                        |
c  |------------------------------------------------------------|
c  |  derived from disper.f, a subroutine to compute            |
c  |           rayleigh wave dispersion curves                  |
c  |           for use with Scilab or Matlab                    |
c  |------------------------------------------------------------|
c  |    nlay=number of layers                                   |
c  |    rho=vector of densities (one for each layer)            |
c  |    mu=vector of shear moduli                               |
c  |    lame=vector of lame's moduli                            |
c  |    zi=vector of layer tops                                 |
c  |    zstart=rigid depth, to start integration                |
c  |    deltz=depth increment for integration                   |
c  |    nfreq=number of frequencies                             |
c  |    flo=lowest frequency                                    |
c  |    delf=frequency increment for dispersion                 |
c  |                                                            |
c  |------------------------------------------------------------|
c  |  dimensions:                                               |
c  |    ndim= number of control points in the model             |
c  |    nfmax= maximum number of frequencies                    |
c  |    maxmod= maximum number of modes                         |
c  |------------------------------------------------------------|
c  |  subroutines in line:                                      |
c  |    matmlt....cvel....objfun....msvctr....                  |
c  |    gparms....halfsp....dmodel....intgr4....pmatr2....      |
c  |    propgt....                                              |
c  |                                                            |
c  |------------------------------------------------------------|
c
      subroutine rwv(nlay,rho,mu,lame,zi,deltz,freq,maxmod,pv)
      implicit real*8 (a-h,o-z)
      real*8 rho(*),mu(*),lame(*),zi(*)
      real*8 rv1(5),rv2(5)
      real*8 rtotal(5)
      real*8 pv(10)
      zero=0.0000000000000000
      epsk=.00000000000001
c...The bracket search requires that jsmax be large enough so
c   that delk is small enough for a successful determination
c   of the phase velocity and wavenumber for each mode.
      jsmax=100
c...See the subroutine cvel.  Failure to bracket the search
c   results when jsmax too small. This is not a problem now
c   with adaptive resetting of jsmax when a bracket failure
c   is about to happen.
      ksw=0
      iskip=1
      pvlcty=0.0
      pfreq=1.0
      modemx=9
      itmax=10
      pi=3.1415927
      pi2=2.00000000*pi
      i5=5
      ndim=nlay
c
c...trim deltz
      nz=int(zi(nlay)/deltz + 0.50)
      deltz=zi(nlay)/dfloat(nz)
c
c...radian/s frequency
      wfreq=2.0*pi*freq
c
c...compute phase velocity
      do 11 jv=1,maxmod
   11 pv(jv)=0.0d+00
      modemx=maxmod
      call cvel(wfreq,pv,jsmax,epsk,modemx,
     +lame,mu,rho,zi,nlay,rv1,rv2,
     +deltz,rtotal,ndim)
c
c...number of modes found
      nmode=0
      do 501 jm=1,maxmod
  501 if(pv(jm).ne.zero) nmode=jm
c
      end
c# |------------------------------------------------------------|
c  | disper.f                                date:  01 may 93   |
c  | author: p. michaels                                        |
c  |------------------------------------------------------------|
c  | matmlt:  matrix multiply                                   |
c  |                                                            |
c  |------------------------------------------------------------|
      subroutine matmlt(a,x,y,n,m)
      implicit real*8 (a-h,o-z)
      real*8 a(n,m),x(m),y(n)
      do 5 j=1,n
      y(j)=0.00d+00
      do 6 k=1,m
    6 y(j)=y(j)+a(j,k)*x(k)
    5 continue
      return
      end
c# |------------------------------------------------------------|
c  | cvel.f                                 date:  01 may 93    |
c  | author: p. michaels                                        |
c  |------------------------------------------------------------|
c  | cvel: calculates phase velocity for full model             |
c  |                                                            |
c  |                                                            |
c  |------------------------------------------------------------|
c
      subroutine cvel(wfreq,pv,jsmax,epsk,modemx,
     +lame,mu,rho,zi,nlay,rv1,rv2,
     +deltz,rtotal,ndim)
      implicit real*8 (a-h,o-z)
      real*8 rho(*),mu(*),lame(*),zi(*)
      real*8 rv1(*),rv2(*)
      real*8 rtotal(*)
      real*8 kstart
      real*8 pv(*)
c
      pi=3.1415927
      iskip=1
c
      iter=0
c
c
c...assign elastic parameters of halfspace
      grho=rho(nlay)
      gmu=mu(nlay)
      glame=lame(nlay)
      betahf=dsqrt(gmu/grho)
c
c...find max and min beta
      betamx=-1.e+10
      betamn=+1.e+10
      do 50 j=1,nlay
      beta=dsqrt(mu(j)/rho(j))
      if(betamx.lt.beta) betamx=beta
   50 if(betamn.gt.beta) betamn=beta
c
      betak=betamn*0.80
      testbm=betahf-betamx
      if(testbm.lt.-0.1) then
ccc      write(10,'(''  !!! warning half space may be too slow'')')
ccc      write(*,'(''  !!! warning half space may be too slow'')')
ccc      write(10,'('' beta max='',f10.4)') betamx
ccc      write(10,'(''  beta half space='',f10.4)') betahf
ccc      write(*,'('' beta max='',f10.4)') betamx
ccc      write(*,'(''  beta half space='',f10.4)') betahf
ccc      close(10)
ccc      stop
      pv(1)=-7777
      return
      betamx=betahf
      endif
c
c...test modemx=maximum number of modes
      if(modemx.gt.9) then
c     write(10,'(''  ****abort**** too many modes'')')
c     write(*,'(''  ****abort**** too many modes'')')
c     close(10)
c     stop
      pv(1)=-8888
      return
      endif
c
c...save jsmax if adapted bracket search compiled version
      jsmax2=jsmax
c
c...loop up to 9 modes
      modect=0
      do 100 jv=1,modemx
c     return to 29 if no cvel found this mode, increased jsmax
c     (return to 29 only activated when iadapt=1)
      iadapt=1
      iacnt=0
   29 continue
c...bail out if adaptive bracket fails in 10 attempts
      if (iacnt .gt. 10) go to 31
c
c...set kstart, and endk
c...note: betak changes at end of loop
c...after each mode is found
      endk=wfreq/(betamx-.00001)
      kstart=wfreq/(betak)
c
      fkh=wfreq/(betak+.1)
      delk=(endk-kstart)/float(jsmax+1)
      fkl=kstart
      ccvl=wfreq/fkh
c
c...test to avoid exceeding betamx
      if(fkl.le.endk) go to 99
c
      call objfun(phil,fkl,iskip,
     +lame,mu,rho,zi,nlay,rv1,rv2,wfreq,
     +deltz,rtotal,ndim,i5)
      jsgn=int(sign(1.0D+00,phil))
c
      do 30 js=1,jsmax
c...test to avoid exceeding betamx
      if(fkh.le.endk) go to 99
c
      call objfun(phih,fkh,iskip,
     +lame,mu,rho,zi,nlay,rv1,rv2,wfreq,
     +deltz,rtotal,ndim,i5)
      jhsgn=int(sign(1.0D+00,phih))
      cphvel=wfreq/fkh
      if(jhsgn.ne.jsgn) go to 40
c
c...move low bracket
      fkl=fkh
c
      fkh=fkh+delk
   30 continue
c...possible bracket search failure, increase jsmax
      if (iadapt .eq. 1) then
      jsmax=jsmax*2
      iacnt=iacnt+1
      go to 29
      endif
   31 continue
c... fatal failure to find a bracket 
      pv(jv)=0.0
      go to 105
   40 continue
c...bracket found, reset jsmax
      jsmax=jsmax2
c
c...capture sign, hi and low k
      isnh=int(sign(1.0D+00,phih))
      isnl=int(sign(1.0D+00,phil))
c
      if(isnl.eq.isnh) then
      pv(1)=-9999
      return
      endif
c
c...find wavenumber and phase velocity
   10 if(iter.gt.1000) then
c      write(*,'(''  max out iterations in cvel '',
c     +'' ( resolution not achieved)'')')
c      write(10,'(''  max out iterations in cvel '',
c     +'' ( resolution not achieved)'')')
      testk=wfreq/(-1111)
      go to 20
      endif
      testk=(fkl+fkh)/2.0
c      write(*,'('' testk='',e14.7)') testk
c
c...avoid exceeding betamx
      if(testk.le.endk) go to 99
c
      if(abs(fkh-fkl).lt.epsk) go to 20
      call objfun(phit,testk,iskip,
     +lame,mu,rho,zi,nlay,rv1,rv2,wfreq,
     +deltz,rtotal,ndim,i5)
      isnt=int(sign(1.0D+00,phit))
c
      if(isnt.eq.isnl) then
      fkl=testk
      isnl=isnt
      else
      fkh=testk
      isnh=isnt
      endif
c
      iter=iter+1
      go to 10
   20 continue
      phasev=wfreq/testk
      pv(jv)=phasev
      betak=phasev + 1.00e+00
      modect=modect+1
      if(modect.eq.modemx) go to 99
c
  100 continue
   99 continue
c----end mode search
  105 continue
c
      return
      end
c# |------------------------------------------------------------|
c  | objfun.f                               date:  01 may 93    |
c  | author: p. michaels                                        |
c  |------------------------------------------------------------|
c  | objfun: calculates objective function                      |
c  |                                                            |
c  |                                                            |
c  |------------------------------------------------------------|
c
      subroutine objfun(phi,fkn1,iskip,
     +lame,mu,rho,zi,nlay,rv1,rv2,wfreq,
     +deltz,rtotal,ndim,i5)
      implicit real*8 (a-h,o-z)
      real*8 rho(*),mu(*),lame(*),zi(*)
      real*8 rv1(5),rv2(5)
      real*8 rtotal(5)
c
      zero=0.00d+00
      mapmat=0
c
      z1=zi(nlay)
c
c...get elastic parameters
      grho=rho(nlay)
      gmu=mu(nlay)
      glame=lame(nlay)
c
c...use trial k to propagate
      phasev=wfreq/fkn1
c
c...set inital p and s wave balance
      aa1=1.0d+00
c
      call halfsp(grho,gmu,glame,wfreq,z1,phasev,
     +rz1,rz2,rz3,rz4,aa1,rv1,rv2)
c
      zstart=zi(nlay)
      rv1(5)=zi(nlay)
      rv1(5)=zi(nlay)
c
c...integrate upwards
      fsign=-1.0
      iprnt=0
      call intgr4(fkn1,zstart,deltz,nlay,wfreq,
     +rtotal,fsign,rho,mu,lame,zi,rv1,rv2,iskip,
     +aa1,iprnt)
c
c...compute objective function
      phi=rv1(3)*rv2(4)-rv2(3)*rv1(4)
c
      return
      end
c# |------------------------------------------------------------|
c  | halfsp.f                               date:  01 may 93    |
c  | author: p. michaels                                        |
c  |------------------------------------------------------------|
c  | halfsp: calculates the theoretical phase velocity and      |
c  |         displacement,stress eigenfunctions at depth, z.    |
c  |                                                            |
c  | rtotal as follows:                                         |
c  | r1=horizontal displacement      (aki & richards sign       |
c  | r2=vertical displacement         convention)               |
c  | r3=horizontal stress                                       |
c  | r4=vertical stress                                         |
c  | aa1= p-wave/s-wave balance ===>  rtotal=aa1*rv1 + rv2      |
c  |       rv1=p-wave set   rv2=s-wave set                      |
c  |                                                            |
c  | c=phase velocity for rayleigh waves                        |
c  |                                                            |
c  | rho=half space density                                     |
c  | fmu=half space shear modulus                               |
c  | flame= half space lame's constant                          |
c  |------------------------------------------------------------|
c
      subroutine halfsp(rho,fmu,flame,wfreq,z,c,r1,r2,r3,r4,
     +aa1,rv1,rv2)
      implicit real*8 (a-h,o-z)
      real*8 rv1(5),rv2(5)
c
c...compute shear velocity
      b=dsqrt(fmu/rho)
      d=flame/fmu
      if(c.eq.0.0e+00) then
c
c   compute phase velocity
c...set up coef. for cubic equation
      a1=-8.0e+00
      a2=((8.0e+00)*(d*3.0e+00+4.0e+00))/(d+2.0e+00)
      a3=((-16.0e+00)*(d+1.0e+00))/(d+2.0e+00)
c
c...solve cubic equation
      pi=3.14159265
      q=(a1*a1-3*a2)/9.0e+00
      r=(2.0e+00*a1*a1*a1-9.0e+00*a1*a2+27.0e+00*a3)/54.0e+00
      test=q*q*q - r*r
      if(test.ge.0.0e+00) then
      t1=acos(r/dsqrt(q*q*q))
      t2=-2.0e+00*dsqrt(q)
      t3=3.0e+00
      x1=t2*cos(t1/t3)-a1/t3
      x2=t2*cos((t1+2.0e+00*pi)/t3)-a1/t3
      x3=t2*cos((t1+4.0e+00*pi)/t3)-a1/t3
      else
      t1=(dsqrt(r*r-q*q*q)+abs(r))**.333333333
      t3=3.0e+00
      x1=-sign(1.0D+00,r)*(t1+q/t1)-a1/t3
      endif
      c=dsqrt(b*b*x1)
      endif
c
c...compute eigenfunctions
      fk=wfreq/c
      t1=(c*c)/(b*b)
      g1=dsqrt(1.0-t1)*fk
      g2=dsqrt(1.0-t1/(d+2.0e+00))*fk
      t2=t1-2.0e+00
      e1=exp(-g1*z)
      e2=exp(-g2*z)
c
c...p-wave set
      rv1(1)=fk*e2
      rv1(2)=g2*e2
      rv1(3)=fmu*fk*(-2.0e+00)*g2*e2
      rv1(4)=(fk*fk*fmu*t2)*e2
      rv1(5)=z
c
c...s-wave set
      rv2(1)=fk*(t2/2.0e+00)*e1
      rv2(2)=((fk*fk)/(2.0e+00*g1))*t2*e1
      rv2(3)=fmu*fk*((t2*t2*fk*fk)/(g1*2.0e+00))*e1
      rv2(4)=-(fk*fk*fmu*t2)*e1
      rv2(5)=z
c
      r1=aa1*fk*e2 + fk*(t2/2.0e+00)*e1
c
      r2=aa1*g2*e2 + ((fk*fk)/(2.0e+00*g1))*t2*e1
c
      r3=fmu*fk*(aa1*(-2.0e+00)*g2*e2  +
     +((t2*t2*fk*fk)/(g1*2.0e+00))*e1)
c
      r4=(fk*fk*fmu*t2)*(aa1*e2 - e1)
c
c
      return
      end
c# |------------------------------------------------------------|
c  |                                         date: 01 may 93    |
c  | author: p. michaels                                        |
c  |------------------------------------------------------------|
c  | intgr4: propagator matrix version of intgrt                |
c  |                                                            |
c  |                                                            |
c  |------------------------------------------------------------|
c
      subroutine intgr4(fkn1,zstart,deltz,nlay,wfreq,
     +rtotal,fsign,rho,mu,lame,zi,rv1,rv2,iskip,
     +aa1,iprnt)
      implicit real*8 (a-h,o-z)
      real*8 rho(*),mu(*),lame(*),zi(*)
      real*8 rv1(*),rv2(*)
      real*8 rtotal(*)
      real*8 p(4,4)
c
      zero=0.0000000000000000
      jzsw=1
      itrig=1
c
c...get elastic parameters
      grho=rho(nlay)
      gmu=mu(nlay)
      glame=lame(nlay)
c
c...use trial k for analytic formula
      phasev=wfreq/fkn1
c
c...upper and lower wavenumbers
c
c...depth integration loop
      if(zstart.ge.zi(nlay)) then
      nz=int(zstart/deltz + 0.50)
      rv1(5)=zstart
      else
      nz=int(zi(nlay)/deltz + 0.50)
      rv1(5)=zi(nlay)
      endif
c
c     write(*,'(''  zstart,deltz,nz'',2e10.4,i5)') zstart,deltz,nz
      lcount=61
      if(nz.lt.1) go to 21
      h=fsign*deltz
      do 20 jz=1,nz
c
c...integrate above zi(nlay), analytic below
c   for stability (as opposed to accuracy)
      test=zi(nlay)
      if(rv1(5).le.test) then
c
c...synchronize integration on zi(nlay)
      if(itrig.eq.1) then
      zin=zi(nlay)
      call halfsp(grho,gmu,glame,wfreq,zin,phasev,
     +rz1,rz2,rz3,rz4,aa1,rv1,rv2)
      itrig=0
      endif
c
c...propagate set 1 and set 2
      z0=rv1(5)
      z1=rv1(5) + h
      call propgt(zi,mu,lame,rho,wfreq,phasev,nlay,
     +p,rv1,rv2,z0,z1)
c
c...analytic eigenfunctions ................................
      else
      z1=rv1(5)+deltz*fsign
      rv1(5)=z1
c
      if(mod(jz,iskip).eq.0 .or. jz.eq.1) then
      call halfsp(grho,gmu,glame,wfreq,z1,phasev,r1,r2,r3,r4,
     +aa1,rv1,rv2)
      endif
c
      endif
c
      if (iprnt.ne.zero) then
      if(mod(jz,iskip).eq.0 .or. jz.eq.1) then
      do 30 jjj=1,4
   30 rtotal(jjj)=aa1*rv1(jjj) + rv2(jjj)
      rtotal(5)=rv1(5)
c
c
      endif
c
c. . . . . . . . . . . . . . . . . . . . . .
c
c...end of block if, depth<zstart
      endif
c
   20 continue
c===(end of depth loop, jz=1,nz)===
c
   21 continue
c
      return
      end
c  |------------------------------------------------------------|
c  | pmatr2.f                               date: 01 may 93     |
c  | author: p. michaels                                        |
c  |------------------------------------------------------------|
c  |                                                            |
c  | pmatrx: compute propagation matrix for a constant layer    |
c  |                                                            |
c  |                                                            |
c  |------------------------------------------------------------|
c
      subroutine pmatr2(p,gmu,glame,grho,wfreq,fkn,
     +z0,z1)
      implicit real*8 (a-h,o-z)
c
      real*8 p(4,4),m,lm,k,z1,z0,r,w
      zero=0.00d+00
      f2=2.00d+00
c
c...get elastic parameters
      m=gmu
      lm=glame
      r=grho
      w=wfreq
      k=fkn
      dz=z1-z0
      a2=(lm+f2*m)/r
      b2=m/r
c
c...test for harmonic solution
      arg1=k*k-(w*w)/(b2)
      arg2=k*k-(w*w)/(a2)
      if(arg1.lt.zero .or. arg2.lt.zero) then
c
c========start g1 complex, g2 real block
      if(arg1.lt.zero .and. arg2.ge.zero) then
      g1=dsqrt(dabs(arg1))
      if(arg2.eq.zero) then
      g2=zero
      else
      g2=dsqrt(arg2)
      endif
      sg1=sin(g1*dz)
      cg1=cos(g1*dz)
      sg2=sinh(g2*dz)
      cg2=cosh(g2*dz)
c
      a=(-k/g1)
      c=(m*(-g1*g1 + k*k))/g1
      e=-f2*k*m
      b=-g2/k
      d=f2*m*g2
      f=(k*k*lm - (lm+f2*m)*g2*g2)/k
      fme=f-e
      adcb=a*d-c*b
c
      p(1,1)=(f*cg1-e*cg2)/fme
      p(3,3)=p(1,1)
c
      p(1,2)=-(c*sg2+d*sg1)/adcb
      p(4,3)=-p(1,2)
c
      p(1,3)=(a*sg2+b*sg1)/adcb
c
      p(1,4)=-(cg1-cg2)/fme
      p(2,3)=-p(1,4)
c
      p(2,1)=(a*f*sg1-b*e*sg2)/fme
      p(3,4)=-p(2,1)
c
      p(2,2)=(a*d*cg1-b*c*cg2)/adcb
      p(4,4)=p(2,2)
c
      p(2,4)=-(a*sg1-b*sg2)/fme
c
      p(3,1)=(c*f*sg1-e*d*sg2)/fme
c
      p(3,2)=(c*d)*((cg1-cg2)/adcb)
      p(4,1)=-p(3,2)
c
      p(4,2)=-(c*f*sg2+e*d*sg1)/adcb
c
      endif
c...============>end complex g1, real g2 block
c
c
c...============>totaly harmonic block
c....both g1 and g2 complex
      if(arg1.lt.zero .and. arg2.lt.zero) then
      g2=dsqrt(dabs(arg2))
      g1=dsqrt(dabs(arg1))
      sg2=sin(g2*dz)
      sg1=sin(g1*dz)
      cg2=cos(g2*dz)
      cg1=cos(g1*dz)
c
      a=(-k/g1)
      c=(m*(-g1*g1 + k*k))/g1
      e=-f2*k*m
      b=-g2/k
      d=f2*m*g2
      f=(k*k*lm + (lm+f2*m)*g2*g2)/k
      fme=f-e
      adcb=a*d-c*b
c
      p(1,1)=(f*cg1-e*cg2)/fme
      p(3,3)=p(1,1)
c
      p(1,2)=-(c*sg2+d*sg1)/adcb
      p(4,3)=-p(1,2)
c
      p(1,3)=(a*sg2+b*sg1)/adcb
c
      p(1,4)=-(cg1-cg2)/fme
      p(2,3)=-p(1,4)
c
      p(2,1)=(a*f*sg1+b*e*sg2)/fme
      p(3,4)=-p(2,1)
c
      p(2,2)=(a*d*cg1-b*c*cg2)/adcb
      p(4,4)=p(2,2)
c
      p(2,4)=-(a*sg1+b*sg2)/fme
c
      p(3,1)=(c*f*sg1+e*d*sg2)/fme
c
      p(3,2)=(c*d)*((cg1-cg2)/adcb)
      p(4,1)=-p(3,2)
c
      p(4,2)=-(c*f*sg2+e*d*sg1)/adcb
c
      endif
c...==============>end totaly harmonic (g1 and g2 complex)

c
      else
c
c...============>totaly non harmonic block
      if(arg2.eq.zero) then
      g2=zero
      else
      g2=dsqrt(arg2)
      endif
      if(arg1.eq.zero) then
      g1=zero
      else
      g1=dsqrt(arg1)
      endif
      sg2=sinh(g2*dz)
      sg1=sinh(g1*dz)
      cg2=cosh(g2*dz)
      cg1=cosh(g1*dz)
c
c      write(10,'(/''  alpha,beta,w,g1,g2 '',5(e10.4,1x))')
c     +a2,b2,w,g1,g2
c
      a=(-k/g1)
      c=(m*(g1*g1 + k*k))/g1
      e=-f2*k*m
      b=-g2/k
      d=f2*m*g2
      f=(k*k*lm - (lm+f2*m)*g2*g2)/k
      fme=f-e
      adcb=a*d-c*b
c
      p(1,1)=(f*cg1-e*cg2)/fme
      p(3,3)=p(1,1)
c
      p(1,2)=-(c*sg2-d*sg1)/adcb
      p(4,3)=-p(1,2)
c
      p(1,3)=(a*sg2-b*sg1)/adcb
c
      p(1,4)=-(cg1-cg2)/fme
      p(2,3)=-p(1,4)
c
      p(2,1)=(a*f*sg1-b*e*sg2)/fme
      p(3,4)=-p(2,1)
c
      p(2,2)=(a*d*cg1-b*c*cg2)/adcb
      p(4,4)=p(2,2)
c
      p(2,4)=-(a*sg1-b*sg2)/fme
c
      p(3,1)=(c*f*sg1-e*d*sg2)/fme
c
      p(3,2)=(c*d)*((cg1-cg2)/adcb)
      p(4,1)=-p(3,2)
c
      p(4,2)=-(c*f*sg2-e*d*sg1)/adcb
c
      endif
c...==============>end totaly non harmonic block
      return
      end
c# |------------------------------------------------------------|
c  | propgt.f                               date:  01 may 93    |
c  | author: p. michaels                                        |
c  |------------------------------------------------------------|
c  |                                                            |
c  | propgt: propagate the motion/stress vector                 |
c  |         from depth z0 to depth z1                          |
c  |                                                            |
c  |------------------------------------------------------------|
      subroutine propgt(zi,mu,lame,rho,wfreq,phasev,nlay,
     +p,rv1,rv2,z0,z1)
      implicit real*8 (a-h,o-z)
      real*8 mu(*),lame(*),rho(*),zi(*)
      real*8 tmp(4),tmp2(4),rv1(5),rv2(5)
      real*8 p(4,4)
c
      fkn=wfreq/phasev
c
      depth=(z0+z1)/2.00d+00
c
c...get elastic parameters
      call gparms(rho,mu,lame,zi,nlay,
     +depth,grho,gmu,glame)
c
      call pmatr2(p,gmu,glame,grho,wfreq,fkn,
     +z0,z1)
c
c      write(10,'('' p'',4(e10.4,1x))') (p(1,j),j=1,4)
c      write(10,'('' p'',4(e10.4,1x))') (p(2,j),j=1,4)
c      write(10,'('' p'',4(e10.4,1x))') (p(3,j),j=1,4)
c      write(10,'('' p'',4(e10.4,1x))') (p(4,j),j=1,4)
      do 5 j=1,4
    5 tmp(j)=rv1(j)
      call matmlt(p,tmp,tmp2,4,4)
      do 10 j=1,4
   10 rv1(j)=tmp2(j)
      rv1(5)=z1
      do 15 j=1,4
   15 tmp(j)=rv2(j)
      call matmlt(p,tmp,tmp2,4,4)
      do 20 j=1,4
   20 rv2(j)=tmp2(j)
      rv2(5)=z1
      return
      end
c# |------------------------------------------------------------|
c  |                                         date: 01 may 93    |
c  | author: p. michaels                                        |
c  |------------------------------------------------------------|
c  | gparms: gets elastic parameters                            |
c  |                                                            |
c  |                                                            |
c  |------------------------------------------------------------|
c
      subroutine gparms(rho,mu,lame,zi,nlay,
     +depth,grho,gmu,glame)
      implicit real*8 (a-h,o-z)
      real*8 mu(*),lame(*),rho(*),zi(*)
c
c...determine current layer
      l1=1
      do 10 j=1,nlay
   10 if(depth.gt.zi(j)) l1=j
c
      if(l1.lt.nlay) then
      l2=l1+1
      dz=zi(l2) - zi(l1)
      sloper=(rho(l2)-rho(l1))/dz
      slopel=(lame(l2)-lame(l1))/dz
      slopem=(mu(l2)-mu(l1))/dz
      z=depth-zi(l1)
      grho=sloper*z + rho(l1)
      gmu=slopem*z + mu(l1)
      glame=slopel*z + lame(l1)
      else
      grho=rho(nlay)
      gmu=mu(nlay)
      glame=lame(nlay)
      endif
      return
      end

