%   |------------------------------------------------------------|
%   | $Id: vfitw.m,v 1.1.1.1 2019/04/29 22:51:20 pm Exp $            |
%   | Basic Seismic Utilities (BSU)                              |
%   | LANGUAGE: octave                                           |
%   | AUTHOR: P. MICHAELS                                        |
%   | FUNCTION NAME: vfitw                                       |
%   | REQUIRES: segyinfo.m and bsegin.m                          |
%   | DIRECTORY: BSUROOT/Octave                                  |
%   |  revised for OCTAVE                                        |
%   |------------------------------------------------------------|
%   | purpose: read a BSEGY file and fit inteval velocities      |
%   |               to vertical times.                           |
%   |------------------------------------------------------------|
%   | octave is an interpreter useful for scientifc calculations.|
%   | It is similar to MATLAB and may be obtained free from      |
%   | http://www.gnu.org/software/octave                         |
%   |------------------------------------------------------------|
%   |  NOTICE:                                                   |
%   |  Copyright (c) 2017 Paul Michaels                          |
%   |  <pm@cgiss.boisestate.edu>                                 |
%   |  This program is free software; you can                    |
%   |  redistribute it and/or modify it under the terms          |
%   |  of the GNU General Public License as published            |
%   |  by the Free Software Foundation; either version           |
%   |  3 of the License, or (at your option) any later           |
%   |  version.  This program is distributed in the              |
%   |  hope that it will be useful, but                          |
%   |  WITHOUT ANY WARRANTY; without even the implied            |
%   |  warranty of MERCHANTABILITY or FITNESS FOR A              |
%   |  PARTICULAR PURPOSE.  See the GNU General Public           |
%   |  License for more details.                                 |
%   |  You should have received a copy of the GNU                |
%   |  General Public License along with this program;           |
%   |  if not, write to the Free Software Foundation,            |
%   |  Inc., 675 Mass Ave, Cambridge, MA 02139, USA.             |
%   |------------------------------------------------------------|
clear

drawnow()
filen=input('enter file name ','s');
fp1=fopen(filen,'r');
%read *.seg file info:  ntraces=number of traces
%                       npts=number samples per trace
%                       fsamin=sample interval in seconds
[nbytes,ntraces,npts,fsamin]=segyinfo(fp1);
%read trace number jtrace by call to bsegin
%        data=vector of digitized samples
%        frange=source receiver separation
%        pic=first arrival time pick if any
%        rx,ry,rz=geophone coordinates
%        sx,sy,sz=source coordinates
%        sd=source depth, below sz
fsamintxt=sprintf(' dt=%.5f sec',fsamin);
xstr=sprintf('%d traces tmax=%.3f sec.',ntraces,npts*fsamin);
disp(fsamintxt)
disp(xstr)

for k=1:ntraces
[data,frange,pic,rxx,ryy,rzz,sxx,syy,szz,sdd]=bsegin(fp1,npts,k);
rx(k)=rxx;
ry(k)=ryy;
rz(k)=rzz;
sx(k)=sxx;
sy(k)=syy;
sz(k)=szz;
sd(k)=sdd;
pics(k)=pic;
%compute vertical time
tvert(k)=pics(k)*(abs(rz(k)-sz(k))/frange);
end;  %next k
fclose(fp1);
%fttxt=sprintf('units = feet ');
%mttxt=sprintf('units = meters ');
%disp(fttxt);
%disp(mttxt);
%fflush(stdout)
%nc=menu('Choose Units',fttxt,mttxt);
nc=questdlg('Choose Units','Units','Feet','Meters','Meters')
switch nc
  case {'Feet'}
  isw2=1;
  case {'Meters'}
  isw2=2;
end

%-------------------Plot Scales---------------------------
fac=.3048;  % f/m scale factor
%nc=questdlg('Choose Maximum Time','Tmax','.025 sec','.100 sec','.100 sec')
%switch nc
%  case{'.025 sec'}
%    xmax=.025;
%  case{'.100 sec'}
%    xmax=.100;
%endswitch


% choose vertical axis type
%nc=menu('Choose Vertical Axis','Elevation','Depth');
nc=questdlg('Choose Vertical Axis','Vert. Axis','Depth','Elevation','Elevation')
switch nc
   case{'Elevation'}
    elev=0;
   case{'Depth'}
    elev=sz(1)-sd(1);
end

if elev==0 
ymin=min(rz);
ymax=max(rz);
end

% plot limits 
if isw2==1 
rz=(rz-elev)/fac;
else
rz=rz-elev;
end

xmin=0; xmax=max(pics); ymin=min(rz); ymax=max(rz);  


Titlecell=inputdlg('Enter Plot Title ','Title',1);
Title1=strjoin(Titlecell);

nc=questdlg('Save to Disk?','Vertical Times','Yes','No','Yes');
switch nc
   case{'Yes'}
    savedata=1;
    disp('writing to disk . . .');
    fp2=fopen(strcat(filen,'.vt'),'w');
      for k=1:ntraces
      fprintf(fp2,' %14.7e %14.7e \n',rz(k),tvert(k));
      end      
    fclose(fp2)
   case{'No'}
    savedata=0;
    disp('skip writing to disk . . .');
end

%Do an interval fit ?
nc=questdlg('Do an interval?','Fit Times','Yes','No','Yes');
switch nc
   case{'Yes'}
    isw1=1;
   case{'No'}
    isw1=0;
end

icnt=0;
plot(tvert,rz,'o')
%axis([xmin,xmax,ymin,ymax])
grid on
xlabel('Vertical Time (s)')
if isw2 ==1
ylabel('Feet');
else
ylabel('Meters');
end
title(Title1);

if savedata==0
  disp('skip writing times to disk .  .  .');
else
%open file to save fits
    fp2=fopen(strcat(filen,'.vt2'),'w');
%write plot parameters to save fit file
  fprintf(fp2,'%s \n',Title1);
  fprintf(fp2,'  %5.5d\n',isw2);
  fprintf(fp2,'  %14.7e  %14.7e  %14.7e  %14.7e\n',xmin,ymin,xmax,ymax);
end;  %endif

% ================================Start Loop==========
% start loop
while isw1==1
icnt=icnt+1;

% select a portion of the data with mouse for linear regression
[x,y]=ginput(2);
zmin=min(y);
zmax=max(y);
npts=length(rz);
[z1,jmin]=min(abs(rz-zmin));
[z2,jmax]=min(abs(rz-zmax));
if jmin>jmax
jhold=jmax;
jmax=jmin;
jmin=jhold;
end

% load interval into yy
k=1;
clear xx; clear yy; clear yycal;
for j=jmin:jmax
yy(k)=tvert(j);
xx(k)=rz(j);
k=k+1;
end

% compute internal sums needed for velocity std.dev.
nptf=length(yy);
sumx=0;
sumy=0;
sumxy=0;
sumx2=0;
for j=1:nptf
sumx=sumx+xx(j);
sumy=sumy+yy(j);
sumxy=sumxy+xx(j)*yy(j);
sumx2=sumx2+xx(j).^2;
end
summ=0;
for j=1:nptf
summ=summ+(sumx-nptf*xx(j)).^2;
end

% linear regression
XX=[ones(nptf,1),xx'];
p=XX\yy' ;

% velocity for interval
vf(icnt)=abs(1/p(2));

% compute calculated data for fit
yycal=XX*p;

%compute time standard deviation (for error bar)
sumd=0;
for j=1:nptf
sumd=sumd+((yy(j)-yycal(j)).^2);
end

sigmay=sqrt(sumd/(nptf));

% compute velocity standard deviation
sigmav=-((sigmay)*sqrt(summ)*((sumx).^2-nptf*sumx2)/((sumx*sumy-nptf*sumxy).^2));
vftxt=sprintf('%d',ceil(vf(icnt)));
sigmavtxt=sprintf('%d',ceil(sigmav));
if isw2==2
msg2=strcat(vftxt,' (+/-',sigmavtxt,'m/s)')
else
msg2=strcat(vftxt,' (+/-',sigmavtxt,'f/s)')
end

% plot calculated data segment
hold on
plot(yycal,xx','-r')

[xl,yl]=ginput(1);
text(xl,yl,msg2);
drawnow

% write fit and label to disk
if savedata==0
  disp('skip writing times to disk .  .  .');
else
  fprintf(fp2,'%14.7e  %14.7e \n',xl,yl);
  fprintf(fp2,'%s \n',msg2);
  fprintf(fp2,'%5.5d \n',length(yycal));
   for k=1:length(yycal)
   fprintf(fp2,'  %14.7e  %14.7e \n',xx(k),yycal(k));
   end
end;  % endif


nc=questdlg('Do another interval?','Fit Times','Yes','No','Yes');
switch nc
   case{'Yes'}
    isw1=1;
   case{'No'}
    isw1=0;
end

end  % endwhile

fclose(fp2);


